<?php

declare(strict_types=1);

namespace Drupal\sites_content_overrides\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Entity\EntityInterface;

/**
 * Custom access check for site override edit routes.
 */
final class SiteOverrideEditAccessCheck {

  /**
   * Checks access for editing a site override.
   *
   * Route requirement uses _custom_access on class::access.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The current user account.
   * @param \Drupal\Core\Entity\EntityInterface|null $entity
   *   The upcasted entity parameter.
   * @param string|null $site
   *   The site ID from the route.
   *
   * @return \Drupal\Core\Access\AccessResult
   *   The access result.
   */
  public static function access(AccountInterface $account): AccessResult {
    // @todo Fix logic here and make it reusable for group access check?
    return AccessResult::allowed();
    // Pull from current route to be resilient to varying param names per type.
    $route_match = \Drupal::routeMatch();
    $parameters = $route_match->getParameters();

    // Find the first upcasted entity parameter.
    $entity = NULL;
    foreach ($parameters as $key => $value) {
      if ($value instanceof EntityInterface) {
        $entity = $value;
        break;
      }
    }
    $site = $route_match->getParameter('site');

    if (!$entity instanceof EntityInterface || empty($site)) {
      return AccessResult::forbidden('Missing entity or site parameter.')->addCacheContexts(['route']);
    }

    // Validate the site exists.
    /** @var \Drupal\sites\SitePluginManagerInterface $site_manager */
    $site_manager = \Drupal::service('plugin.manager.site');
    if (!$site_manager->hasDefinition((string) $site)) {
      return AccessResult::forbidden('Invalid site.')->addCacheContexts(['route']);
    }

    // Check capability for this entity.
    /** @var \Drupal\sites\SitesServiceInterface $sites_service */
    $sites_service = \Drupal::service('sites.service');
    if (!$sites_service->entitySupportsSiteContent($entity)) {
      return AccessResult::forbidden('Entity type not supported for site overrides.')->addCacheableDependency($entity);
    }

    // Delegate to core entity update access; also handled by _entity_access on the route.
    $update_access = $entity->access('update', $account, TRUE);

    return AccessResult::allowedIf($update_access->isAllowed())
      ->addCacheableDependency($update_access)
      ->addCacheableDependency($entity)
      ->addCacheContexts(['user.permissions']);
  }

}
