<?php

declare(strict_types=1);

namespace Drupal\sites_content_overrides\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\sites_content_overrides\SiteAwareEntityResolverInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Demo controller rendering the effective site-specific revision.
 */
final class OverrideViewController extends ControllerBase implements ContainerInjectionInterface {

  public function __construct(
    private readonly SiteAwareEntityResolverInterface $resolver,
    private readonly RendererInterface $renderer,
  ) {}

  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('sites_content_overrides.site_aware_entity_resolver'),
      $container->get('renderer'),
    );
  }

  /**
   * Renders the effective revision for the given node entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to render.
   * @param string $view_mode
   *   The view mode to use.
   *
   * @return array
   *   A render array for the effective revision.
   */
  public function view(ContentEntityInterface $entity, string $view_mode = 'full'): array {
    $effective = $this->resolver->loadEffective($entity);
    if (!$effective instanceof ContentEntityInterface) {
      throw new NotFoundHttpException();
    }

    $view_builder = $this->entityTypeManager()->getViewBuilder($effective->getEntityTypeId());
    $build = $view_builder->view($effective, $view_mode);

    // Ensure responses vary by site context and depend on the entity revision.
    $build['#cache']['contexts'][] = 'site';
    $this->renderer->addCacheableDependency($build, $effective);

    return $build;
  }

}
