<?php

declare(strict_types=1);

namespace Drupal\sites_content_overrides\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Render\Markup;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Lists site-specific override revisions for an entity.
 */
final class SiteOverrideRevisionsController extends ControllerBase implements ContainerInjectionInterface {

  public function __construct(
    private readonly DateFormatterInterface $dateFormatter,
  ) {}

  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('date.formatter'),
    );
  }

  /**
   * Builds a revisions table filtered by site.
   */
  public function listing(\Drupal\Core\Routing\RouteMatchInterface $route_match): array {
    $entity = NULL;
    foreach ($route_match->getParameters() as $param) {
      if ($param instanceof ContentEntityInterface) {
        $entity = $param;
        break;
      }
    }
    if (!$entity instanceof ContentEntityInterface) {
      throw new NotFoundHttpException();
    }

    $site_id = (string) ($route_match->getParameter('site') ?? '');
    if ($site_id === '') {
      throw new NotFoundHttpException();
    }

    $storage = \Drupal::entityTypeManager()->getStorage($entity->getEntityTypeId());
    if (!$storage instanceof \Drupal\Core\Entity\RevisionableStorageInterface) {
      return ['#markup' => $this->t('This entity type does not support revisions.')];
    }

    $entity_type = $entity->getEntityType();
    $id_key = $entity_type->getKey('id');
    $rev_key = $entity_type->getKey('revision');

    $header = [
      $this->t('Revision'),
      $this->t('Date'),
      $this->t('Author'),
      $this->t('Message'),
      $this->t('Operations'),
    ];

    $query = $storage->getQuery()->allRevisions();
    $query->condition($id_key, $entity->id());
    $query->condition('site_id', $site_id);
    $query->sort($rev_key, 'DESC');
    $query->accessCheck(TRUE);
    $result = $query->execute();

    $rows = [];
    if (!empty($result)) {
      foreach ($result as $revision_id => $_) {
        /** @var \Drupal\Core\Entity\ContentEntityInterface $revision */
        $revision = $storage->loadRevision($revision_id);
        if (!$revision instanceof ContentEntityInterface) {
          continue;
        }

        $is_current = ((int) $entity->getRevisionId() === (int) $revision->getRevisionId());
        $date = $this->dateFormatter->format($revision->getRevisionCreationTime(), 'short');
        $author = $revision->getRevisionUser() ? $revision->getRevisionUser()->toLink() : Link::fromTextAndUrl($this->t('Unknown'), Url::fromRoute('<none>'));
        $message = '';
        if ($revision->hasField('revision_log')) {
          $message_val = (string) $revision->get('revision_log')->value;
          $message = Markup::create(nl2br(htmlspecialchars($message_val)));
        }

        $operations = [
          '#type' => 'operations',
          '#links' => [],
        ];

        // Restore (copy values) operation; opens in modal.
        $restore_url = Url::fromRoute('entity.' . $entity->getEntityTypeId() . '.site_override_restore', [
          $entity->getEntityTypeId() => $entity->id(),
          'site' => $site_id,
          'revision' => $revision_id,
        ]);
        $operations['#links']['restore'] = [
          'title' => $is_current ? $this->t('Re-save as new') : $this->t('Restore'),
          'url' => $restore_url,
          'attributes' => [
            'class' => ['use-ajax'],
            'data-dialog-type' => 'modal',
            'data-dialog-options' => json_encode(['width' => 800]),
          ],
        ];

        $rows[] = [
          'data' => [
            ['data' => ['#markup' => (string) $revision_id . ($is_current ? ' ' . $this->t('(current)') : '')]],
            ['data' => ['#markup' => $date]],
            ['data' => $author->toRenderable()],
            ['data' => ['#markup' => $message]],
            ['data' => $operations],
          ],
        ];
      }
    }

    $build = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No overrides have been created for this site yet.'),
      '#attached' => [
        'library' => ['core/drupal.dialog.ajax'],
      ],
      '#cache' => [
        'contexts' => ['user.permissions', 'site'],
        'tags' => $entity->getCacheTags(),
      ],
    ];
    \Drupal::service('renderer')->addCacheableDependency($build, $entity);

    return $build;
  }
}
