<?php

declare(strict_types=1);

namespace Drupal\sites_content_overrides\Controller;

use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Builds dynamic titles for site override routes.
 */
final class SiteOverrideTitleCallback {
  use StringTranslationTrait;

  /**
   * Title callback for the Edit route.
   */
  public static function title(RouteMatchInterface $route_match): string {
    [$entity_label, $site_label] = self::extractLabels($route_match);
    return (string) \Drupal::translation()->translate('Edit @entity for @site', [
      '@entity' => $entity_label,
      '@site' => $site_label,
    ]);
  }

  /**
   * Title callback for the View route.
   */
  public static function viewTitle(RouteMatchInterface $route_match): string {
    [$entity_label, $site_label] = self::extractLabels($route_match);
    return (string) \Drupal::translation()->translate('Override for @site: @entity', [
      '@entity' => $entity_label,
      '@site' => $site_label,
    ]);
  }

  /**
   * Helper to resolve entity and site labels from the route.
   *
   * @return array{0:string,1:string}
   *   [entity_label, site_label]
   */
  private static function extractLabels(RouteMatchInterface $route_match): array {
    $parameters = $route_match->getParameters();
    $entity = NULL;
    foreach ($parameters as $value) {
      if ($value instanceof \Drupal\Core\Entity\EntityInterface) {
        $entity = $value;
        break;
      }
    }
    $site_id = (string) ($route_match->getParameter('site') ?? '');

    $site_label = $site_id;
    if ($site_id !== '') {
      /** @var \Drupal\sites\SitePluginManagerInterface $site_manager */
      $site_manager = \Drupal::service('plugin.manager.site');
      if ($site_manager->hasDefinition($site_id)) {
        $def = $site_manager->getDefinition($site_id);
        $site_label = is_string($def['label'] ?? NULL) ? $def['label'] : $site_id;
      }
    }

    $entity_label = $entity ? (string) $entity->label() : '';
    return [$entity_label, $site_label];
  }
}
