<?php

declare(strict_types=1);

namespace Drupal\sites_content_overrides\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Url;
use Drupal\sites_content_overrides\SiteAwareEntityResolverInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Controller to view a specific site's override of an entity.
 */
final class SiteOverrideViewController extends ControllerBase implements ContainerInjectionInterface {

  public function __construct(
    private readonly SiteAwareEntityResolverInterface $resolver,
    private readonly RendererInterface $renderer,
  ) {}

  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get(SiteAwareEntityResolverInterface::class),
      $container->get('renderer'),
    );
  }

  /**
   * Renders the effective site override for the given entity and site.
   *
   * Route params are generic: {entity_type_id}, {entity}, {site}.
   */
  public function view(\Drupal\Core\Routing\RouteMatchInterface $route_match): array {
    // Resolve the upcasted entity parameter regardless of its name.
    $entity = NULL;
    foreach ($route_match->getParameters() as $param) {
      if ($param instanceof ContentEntityInterface) {
        $entity = $param;
        break;
      }
    }
    if (!$entity instanceof ContentEntityInterface) {
      throw new NotFoundHttpException();
    }

    // Resolve the site object by ID from route parameter 'site'.
    $site = (string) ($route_match->getParameter('site') ?? '');
    /** @var \Drupal\sites\SitePluginManagerInterface $site_manager */
    $site_manager = \Drupal::service('plugin.manager.site');
    if ($site === '' || !$site_manager->hasDefinition($site)) {
      throw new NotFoundHttpException();
    }
    /** @var \Drupal\sites\SiteInterface $site_obj */
    $site_obj = $site_manager->getSite($site);

    // Load the composed effective entity for this site.
    $effective = $this->resolver->loadEffective($entity, $site_obj);
    if (!$effective instanceof ContentEntityInterface) {
      throw new NotFoundHttpException();
    }

    $view_builder = $this->entityTypeManager()->getViewBuilder($effective->getEntityTypeId());
    $build = $view_builder->view($effective, 'full');

    // Ensure responses vary by site context and depend on the entity revision.
    $build['#cache']['contexts'][] = 'site';
    $this->renderer->addCacheableDependency($build, $effective);

    return $build;
  }

  /**
   * Redirects generic edit route to the concrete per-entity edit route.
   */
  public function redirectToEdit(string $entity_type_id, string|int $entity, string $site): RedirectResponse {
    $route_name = 'entity.' . $entity_type_id . '.site_override_edit';
    $url = Url::fromRoute($route_name, [
      $entity_type_id => $entity,
      'site' => $site,
    ]);
    return new RedirectResponse($url->toString());
  }

}
