<?php

declare(strict_types=1);

namespace Drupal\sites_content_overrides\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Url;
use Drupal\sites\SitesServiceInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;

/**
 * Lists site override options for an entity and links to edit forms.
 */
final class SiteOverridesListController extends ControllerBase implements ContainerInjectionInterface {

  public function __construct(
    protected SitesServiceInterface $sitesService,
  ) {}

  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('sites.service'),
    );
  }

  /**
   * Overview listing for site overrides.
   *
   * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
   *   A render array or redirect to the sole site's edit form.
   */
  public function listing(\Drupal\Core\Routing\RouteMatchInterface $route_match): array|RedirectResponse {
    // Resolve the upcasted entity parameter regardless of its name.
    $entity = NULL;
    foreach ($route_match->getParameters() as $param) {
      if ($param instanceof \Drupal\Core\Entity\ContentEntityInterface) {
        $entity = $param;
        break;
      }
    }
    if (!$entity instanceof \Drupal\Core\Entity\ContentEntityInterface) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    // Ensure user may update the entity; route should enforce, but double-check.
    $access = $entity->access('override', NULL, TRUE);
    if (!$access->isAllowed()) {
      throw new AccessDeniedHttpException();
    }

    // Resolve candidate sites for this entity.
    $candidates = $this->sitesService->getSiteCandiatesForEntityAndUser($entity, \Drupal::currentUser()) ?? [];

    // Auto-redirect when only one site candidate exists.
    $current_site = \Drupal::service('current_site')->getSite();
    if ($current_site->id() != 'default' || count($candidates) === 1) {
      $site = reset($candidates);

      if ($current_site->id() != 'default') {
        $site = $current_site;
      }

      $entity_type_id = $entity->getEntityTypeId();
      $url = Url::fromRoute('entity.' . $entity_type_id . '.site_override_edit', [
        $entity_type_id => $entity->id(),
        'site' => $site->id(),
      ], ['query' => \Drupal::service('redirect.destination')->getAsArray()]);
      // *** Crucial Step: Tell Drupal to ignore the original destination parameter. ***
//      $url->setIgnoreDestination(TRUE);
//      return new RedirectResponse($url->toString());

      $request = \Drupal::request();

      // Temporarily remove the 'destination' parameter from the request query.
      $original_destination = $request->query->get('destination');
      $request->query->remove('destination');

      // Create your intended redirect response.
      $response = new RedirectResponse($url->toString());

      // You can restore the parameter to the request stack afterwards.
      if ($original_destination) {
//        $request->query->set('destination', $original_destination);
      }

      return $response;
    }

    // Build table header.
    $header = [
      $this->t('Site'),
      $this->t('Status'),
      $this->t('Operations'),
    ];

    $rows = [];
    foreach ($candidates as $site_id => $site) {
      /** @var \Drupal\sites_content_overrides\Service\SiteOverrideCounterInterface $counter */
      $counter = \Drupal::service('sites_content_overrides.override_counter');
      $has_override = $counter->hasOverrideForSite($entity, $site_id);

      $status = $has_override ? $this->t('Overridden') : $this->t('Inherited');
      $entity_type_id = $entity->getEntityTypeId();
      $edit_url = Url::fromRoute('entity.' . $entity_type_id . '.site_override_edit', [
        $entity_type_id => $entity->id(),
        'site' => $site_id,
      ]);

      $op_label = $has_override ? $this->t('Edit override') : $this->t('Create override');
      $operations = [
        '#type' => 'operations',
        '#links' => [
          'edit' => [
            'title' => $op_label,
            'url' => $edit_url,
          ],
        ],
      ];

      // Add a Revisions operation linking to the site-specific revisions page.
      $revisions_url = Url::fromRoute('entity.' . $entity_type_id . '.site_override_revisions', [
        $entity_type_id => $entity->id(),
        'site' => $site_id,
      ]);
      $operations['#links']['revisions'] = [
        'title' => $this->t('Revisions'),
        'url' => $revisions_url,
      ];

      // Add a Layout operation if Layout Builder overrides are active for this entity.
      try {
        /** @var \Drupal\Core\Entity\Display\EntityDisplayRepositoryInterface $display_repo */
        $display_repo = \Drupal::service('entity_display.repository');
        $display = $display_repo->getViewDisplay($entity_type_id, $entity->bundle(), 'default');
        if ($display instanceof \Drupal\layout_builder\Entity\LayoutBuilderEntityViewDisplay && $display->isLayoutBuilderEnabled() && $display->isOverridable()) {
          $layout_url = Url::fromRoute('layout_builder.site_overrides.' . $entity_type_id . '.view', [
            $entity_type_id => $entity->id(),
            'site' => $site_id,
          ]);
          $operations['#links']['layout'] = [
            'title' => $this->t('Layout'),
            'url' => $layout_url,
          ];
        }
      }
      catch (\Throwable $e) {
        // If Layout Builder is not enabled, just skip adding the Layout op.
      }

      // Add a Revert operation when an override exists for this site.
      if ($has_override) {
        $revert_url = Url::fromRoute('entity.' . $entity_type_id . '.site_override_revert', [
          $entity_type_id => $entity->id(),
          'site' => $site_id,
        ]);
        $operations['#links']['revert'] = [
          'title' => $this->t('Revert to canonical'),
          'url' => $revert_url,
          'attributes' => [
            'class' => ['use-ajax'],
            'data-dialog-type' => 'modal',
            'data-dialog-options' => json_encode(['width' => 800]),
          ],
        ];
      }

      $rows[] = [
        'data' => [
          ['data' => ['#markup' => $site->label()]],
          ['data' => ['#markup' => $status]],
          ['data' => $operations],
        ],
      ];
    }

    $build = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No sites available for overrides.'),
      '#attached' => [
        'library' => [
          'core/drupal.dialog.ajax',
        ],
      ],
      '#cache' => [
        'contexts' => ['user.permissions'],
        'tags' => $entity->getCacheTags(),
      ],
    ];
    // Ensure entity dependency so changes invalidate the listing.
    \Drupal::service('renderer')->addCacheableDependency($build, $entity);

    return $build;
  }

}
