<?php

declare(strict_types=1);

namespace Drupal\sites_content_overrides\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\sites_content_overrides\SiteAwareEntityResolverInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Confirm form to restore a previous site-override revision as a new revision.
 */
final class RestoreOverrideRevisionConfirmForm extends ConfirmFormBase implements ContainerInjectionInterface {

  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly SiteAwareEntityResolverInterface $resolver,
  ) {}

  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get(SiteAwareEntityResolverInterface::class),
    );
  }

  public function getFormId(): string {
    return 'sites_content_overrides_restore_revision_confirm';
  }

  protected function getRoutedEntity(): ?ContentEntityInterface {
    $route_match = \Drupal::routeMatch();
    foreach ($route_match->getParameters() as $param) {
      if ($param instanceof ContentEntityInterface) {
        return $param;
      }
    }
    return NULL;
  }

  protected function getSiteIdFromRoute(): ?string {
    $site = \Drupal::routeMatch()->getParameter('site');
    return is_string($site) ? $site : NULL;
  }

  protected function getRevisionIdFromRoute(): ?int {
    $rev = \Drupal::routeMatch()->getParameter('revision');
    return is_numeric($rev) ? (int) $rev : NULL;
  }

  public function getQuestion() {
    $entity = $this->getRoutedEntity();
    $site_id = $this->getSiteIdFromRoute() ?? 'unknown';
    $label = $entity ? $entity->label() : '';
    $rev_id = $this->getRevisionIdFromRoute();
    return $this->t('Restore override revision @rev for %label on site %site?', ['@rev' => $rev_id, '%label' => $label, '%site' => $site_id]);
  }

  public function getDescription() {
    return $this->t('This will copy the values from the selected site-specific revision into a new revision, keeping other fields and metadata intact.');
  }

  public function getCancelUrl(): Url {
    $entity = $this->getRoutedEntity();
    $site_id = $this->getSiteIdFromRoute();
    $entity_type_id = $entity?->getEntityTypeId() ?? 'entity';
    return Url::fromRoute('entity.' . $entity_type_id . '.site_override_revisions', [
      $entity_type_id => $entity?->id(),
      'site' => $site_id,
    ]);
  }

  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildForm($form, $form_state);

    $entity = $this->getRoutedEntity();
    $site_id = $this->getSiteIdFromRoute();
    $rev_id = $this->getRevisionIdFromRoute();
    if (!$entity || !$site_id || !$rev_id) {
      $this->messenger()->addError($this->t('Invalid route parameters.'));
      $form['message'] = ['#markup' => $this->t('Invalid route parameters.')];
      return $form;
    }

    $form['summary'] = [
      '#type' => 'item',
      '#markup' => $this->t('A new revision will be created for site %site based on revision @rev.', ['%site' => $site_id, '@rev' => $rev_id]),
    ];

    return $form;
  }

  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $entity = $this->getRoutedEntity();
    $site_id = $this->getSiteIdFromRoute();
    $rev_id = $this->getRevisionIdFromRoute();
    if (!$entity || !$site_id || !$rev_id) {
      $this->messenger()->addError($this->t('Invalid route parameters.'));
      $form_state->setRedirectUrl(Url::fromRoute('<front>'));
      return;
    }

    $storage = $this->entityTypeManager->getStorage($entity->getEntityTypeId());
    if (!$storage instanceof \Drupal\Core\Entity\RevisionableStorageInterface) {
      $this->messenger()->addError($this->t('This entity type does not support revisions.'));
      $form_state->setRedirectUrl($this->getCancelUrl());
      return;
    }

    /** @var \Drupal\Core\Entity\ContentEntityInterface $source */
    $source = $storage->loadRevision($rev_id);
    if (!$source instanceof ContentEntityInterface) {
      $this->messenger()->addError($this->t('The selected revision could not be loaded.'));
      $form_state->setRedirectUrl($this->getCancelUrl());
      return;
    }

    // Create a new revision based on the current working entity, then copy
    // overrideable fields from the selected revision.
    /** @var \Drupal\Core\Entity\ContentEntityInterface $new_revision */
    $new_revision = $storage->createRevision($entity, FALSE);
    if ($new_revision->hasField('site_id')) {
      $new_revision->set('site_id', $site_id);
    }

    $langcode = $new_revision->language()->getId();
    $translation = $new_revision->hasTranslation($langcode) ? $new_revision->getTranslation($langcode) : $new_revision;

    // Copy overrideable fields values from the source revision.
    foreach ($translation->getFieldDefinitions() as $field_name => $definition) {
      $entity_type = $translation->getEntityType();
      if (in_array($field_name, [$entity_type->getKey('revision'), $entity_type->getKey('id'), 'site_id'], TRUE)) {
        continue;
      }
      if (!$this->resolver->isFieldOverrideable($field_name, $translation)) {
        continue;
      }
      if ($source->hasField($field_name)) {
        $translation->set($field_name, $source->get($field_name)->getValue());
      }
    }

    // Mark as NOT translation-affecting and add a log message.
    $translation->setRevisionTranslationAffected(FALSE);
    if ($translation->hasField('revision_log')) {
      $existing = (string) $translation->get('revision_log')->value;
      $msg = $this->t('Restored site override from revision @rev for site %site.', ['@rev' => $rev_id, '%site' => $site_id]);
      $translation->set('revision_log', trim($existing . "\n" . $msg));
    }

    try {
      $translation->save();
      $this->messenger()->addStatus($this->t('A new override revision has been created from revision @rev.', ['@rev' => $rev_id]));
    }
    catch (\Throwable $e) {
      \Drupal::logger('sites_content_overrides')->error('Restore failed: @msg', ['@msg' => $e->getMessage()]);
      $this->messenger()->addError($this->t('Failed to restore the override.'));
    }

    $form_state->setRedirectUrl($this->getCancelUrl());
  }
}
