<?php

declare(strict_types=1);

namespace Drupal\sites_content_overrides\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\sites_content_overrides\SiteAwareEntityResolverInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Confirm form to revert a site override to canonical (remove override values).
 */
final class RevertOverrideConfirmForm extends ConfirmFormBase implements ContainerInjectionInterface {

  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly SiteAwareEntityResolverInterface $resolver,
  ) {}

  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get(SiteAwareEntityResolverInterface::class),
    );
  }

  public function getFormId(): string {
    return 'sites_content_overrides_revert_confirm';
  }

  /**
   * Resolve the routed entity parameter regardless of its name.
   */
  protected function getRoutedEntity(): ?ContentEntityInterface {
    $route_match = \Drupal::routeMatch();
    foreach ($route_match->getParameters() as $param) {
      if ($param instanceof ContentEntityInterface) {
        return $param;
      }
    }
    return NULL;
  }

  protected function getSiteIdFromRoute(): ?string {
    $site = \Drupal::routeMatch()->getParameter('site');
    return is_string($site) ? $site : NULL;
  }

  public function getQuestion() {
    $entity = $this->getRoutedEntity();
    $site_id = $this->getSiteIdFromRoute() ?? 'unknown';
    $label = $entity ? $entity->label() : '';
    return $this->t('Revert override for %label on site %site?', ['%label' => $label, '%site' => $site_id]);
  }

  public function getDescription() {
    return $this->t('This will create a new revision that removes all site-specific override values so the entity inherits values from the canonical revision for this site.');
  }

  public function getCancelUrl(): Url {
    $entity = $this->getRoutedEntity();
    $site_id = $this->getSiteIdFromRoute();
    $entity_type_id = $entity?->getEntityTypeId() ?? 'entity';
    return Url::fromRoute('entity.' . $entity_type_id . '.site_override_edit', [
      $entity_type_id => $entity?->id(),
      'site' => $site_id,
    ]);
  }

  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildForm($form, $form_state);

    // Safety checks.
    $entity = $this->getRoutedEntity();
    $site_id = $this->getSiteIdFromRoute();
    if (!$entity || !$site_id) {
      $this->messenger()->addError($this->t('Invalid route parameters.'));
      $form['message'] = ['#markup' => $this->t('Invalid route parameters.')];
      return $form;
    }

    // Show a short summary of the action.
    $revisions_url = Url::fromRoute('entity.' . $entity->getEntityTypeId() . '.site_override_revisions', [
      $entity->getEntityTypeId() => $entity->id(),
      'site' => $site_id,
    ])->toString();
    $form['summary'] = [
      '#type' => 'item',
      '#markup' => $this->t('A new revision will be created for this entity and site that removes all override values. You can restore previous overrides from the <a href=":url">revision history</a> if needed.', [':url' => $revisions_url]),
    ];

    return $form;
  }

  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $entity = $this->getRoutedEntity();
    $site_id = $this->getSiteIdFromRoute();
    if (!$entity || !$site_id) {
      $this->messenger()->addError($this->t('Invalid route parameters.'));
      $form_state->setRedirectUrl(Url::fromRoute('<front>'));
      return;
    }

    $storage = $this->entityTypeManager->getStorage($entity->getEntityTypeId());
    if (!method_exists($storage, 'createRevision')) {
      $this->messenger()->addError($this->t('This entity type does not support revisions.'));
      $form_state->setRedirectUrl($this->getCancelUrl());
      return;
    }

    // Create a new revision from the current override edit context.
    /** @var \Drupal\Core\Entity\ContentEntityInterface $new_revision */
    $new_revision = $storage->createRevision($entity, FALSE);
    // Ensure the site context is retained.
    if ($new_revision->hasField('site_id')) {
      $new_revision->set('site_id', $site_id);
    }

    // Operate on active translation only.
    $langcode = $new_revision->language()->getId();
    $translation = $new_revision->hasTranslation($langcode) ? $new_revision->getTranslation($langcode) : $new_revision;

    // Remove all override values for fields that are overrideable so the
    // effective entity falls back to canonical values.
    foreach ($translation->getFieldDefinitions() as $field_name => $definition) {
      // Skip internal IDs and site_id.
      $entity_type = $translation->getEntityType();
      $skip = in_array($field_name, [$entity_type->getKey('revision'), $entity_type->getKey('id'), 'site_id'], TRUE);
      if ($skip) {
        continue;
      }
      // Only clear overrideable fields.
      if (!$this->resolver->isFieldOverrideable($field_name, $translation)) {
        continue;
      }
      if ($translation->hasField($field_name)) {
        // Empty the field completely (works for single and multi-value fields).
        $translation->set($field_name, []);
      }
    }

    // Mark as NOT translation-affecting to keep out of default revision listings.
    $translation->setRevisionTranslationAffected(FALSE);

    // Add a revision log message for traceability.
    if ($translation->hasField('revision_log')) {
      $existing = (string) $translation->get('revision_log')->value;
      $msg = $this->t('Reverted site override to canonical for site %site.', ['%site' => $site_id]);
      $translation->set('revision_log', trim($existing . "\n" . $msg));
    }

    try {
      $translation->save();
      $this->messenger()->addStatus($this->t('Override reverted. The entity now inherits values from the canonical revision.'));
    }
    catch (\Throwable $e) {
      \Drupal::logger('sites_content_overrides')->error('Revert failed: @msg', ['@msg' => $e->getMessage()]);
      $this->messenger()->addError($this->t('Failed to revert the override.'));
    }

    // Redirect back to the edit form so user can see inherited values.
    $form_state->setRedirectUrl($this->getCancelUrl());
  }
}
