<?php

declare(strict_types=1);

namespace Drupal\sites_content_overrides\FormDecorator;

use Drupal\Core\Entity\EntityFormInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\field\FieldConfigInterface;
use Drupal\form_decorator\Attribute\FormDecorator;
use Drupal\form_decorator\EntityFormDecoratorBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Adds the "Allow per-site overrides" checkbox to FieldConfig edit forms.
 */
#[FormDecorator()]
final class FieldConfigOverrideableDecorator extends EntityFormDecoratorBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a FieldConfigOverrideableDecorator object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function applies(): bool {
    if (!$this->inner instanceof EntityFormInterface) {
      return FALSE;
    }

    return $this->getEntity() instanceof FieldConfigInterface;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ...$args): array {
    $form = parent::buildForm($form, $form_state, ...$args);

    $entity = $this->getEntity();

    // Only show for entity types that participate in sites and are revisionable
    // content entities.
    $supported_entity_type_ids = sites_supported_entity_type_ids();
    if (!in_array($entity->getTargetEntityTypeId(), $supported_entity_type_ids, true)) {
      return $form;
    }

    $default = (bool) $entity->getThirdPartySetting('sites_content_overrides', 'overrideable', FALSE);
    $form['sites_content_overrides_overrideable'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow per-site overrides'),
      '#description' => $this->t('If checked, this field can have different values per site; otherwise it always uses the canonical value.'),
      '#default_value' => $default,
      '#weight' => 100,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // First, let the inner form handle submission (so values exist in state).
    parent::submitForm($form, $form_state);

    $value = (bool) $form_state->getValue('sites_content_overrides_overrideable');
    $this->getEntity()->setThirdPartySetting('sites_content_overrides', 'overrideable', $value);
  }

}
