<?php

declare(strict_types=1);

namespace Drupal\sites_content_overrides\FormDecorator;

use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\Element;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\form_decorator\ContentEntityFormDecoratorBase;
use Drupal\form_decorator\Attribute\FormDecorator;
use Drupal\sites_content_overrides\Controller\SiteOverrideTitleCallback;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Ensures site-specific revision behavior via form decorator.
 */
#[FormDecorator()]
final class SiteOverrideContentEntityFormDecorator extends ContentEntityFormDecoratorBase implements ContainerFactoryPluginInterface {

  use DependencySerializationTrait;

  /**
   * Constructs a \Drupal\Component\Plugin\PluginBase object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   *   The current route match.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly RouteMatchInterface $routeMatch,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('current_route_match'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function applies(): bool {
    foreach (sites_supported_entity_type_ids() as $entity_type_id) {
      if ($this->routeMatch->getRouteName() == "entity.{$entity_type_id}.site_override_edit") {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ...$args) {
    // Let the original form build first.
    $form = parent::buildForm($form, $form_state, ...$args);

    // @todo Title is not working.
    $form['#title'] = SiteOverrideTitleCallback::title($this->routeMatch);
    // Add an informational message on the form.
    $form['sites_content_overrides_info'] = [
      '#type' => 'container',
      'message' => [
        '#markup' => $this->t('Only fields marked as overrideable can be changed for this site. Other fields are inherited from the canonical version.'),
      ],
      '#weight' => -100,
    ];

    $form['actions']['submit']['#value'] = $this->t('Save override');
    unset($form['actions']['delete']);

    // Add a Revert action that leads to a confirmation form.
    $entity = $this->getEntity();
    $entity_type_id = $entity->getEntityTypeId();
    $site_id = (string) $this->routeMatch->getParameter('site');
    $revert_url = \Drupal\Core\Url::fromRoute('entity.' . $entity_type_id . '.site_override_revert', [
      $entity_type_id => $entity->id(),
      'site' => $site_id,
    ], ['query' => \Drupal::service('redirect.destination')->getAsArray()]);

    $form['actions']['revert'] = [
      '#type' => 'link',
      '#title' => $this->t('Revert to canonical'),
      '#url' => $revert_url,
      '#attributes' => [
        'class' => ['button', 'button--danger'],
      ],
      '#weight' => 50,
    ];

    $form['#process'][] = [$this, 'entityFormSharedElements'];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $entity = $this->getEntity();
    $site_id = $this->routeMatch->getParameter('site');
    $entity->set('site_id', $site_id);

    $this->inner->validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $entity = $this->getEntity();
    $storage = $this->entityTypeManager->getStorage($entity->getEntityTypeId());

    // Create a new revision based on the current edited entity values.
    $entity = $storage->createRevision($entity, FALSE);

    // Mark this override revision as NOT translation-affecting so it does not
    // appear on the default revision overview listings.
    $langcode = $entity->language()->getId();
    $translation = $entity->hasTranslation($langcode) ? $entity->getTranslation($langcode) : $entity;
    $translation->setRevisionTranslationAffected(FALSE);

    return $entity->save();
  }


  /**
   * Process callback: determines which elements get clue in the form.
   *
   * @see \Drupal\content_translation\ContentTranslationHandler::entityFormAlter()
   */
  public function entityFormSharedElements($element, FormStateInterface $form_state, $form) {
    static $ignored_types;

    // @todo Find a more reliable way to determine if a form element concerns a
    //   multilingual value.
    if (!isset($ignored_types)) {
      $ignored_types = array_flip(['actions', 'value', 'hidden', 'vertical_tabs', 'token', 'details', 'link']);
    }

    /** @var \Drupal\Core\Entity\ContentEntityForm $form_object */
    $form_object = $form_state->getFormObject();
    /** @var \Drupal\Core\Entity\ContentEntityInterface $entity */
    $entity = $form_object->getEntity();
    $display_translatability_clue = TRUE;
    $hide_untranslatable_fields = $entity->isDefaultTranslationAffectedOnly() && !$entity->isDefaultTranslation();
    $hide_untranslatable_fields = TRUE;
    $translation_form = $form_state->get(['content_translation', 'translation_form']);
    $display_warning = FALSE;

    // We use field definitions to identify untranslatable field widgets to be
    // hidden. Fields that are not involved in translation changes checks should
    // not be affected by this logic (the "revision_log" field, for instance).
//    $field_definitions = array_diff_key($entity->getFieldDefinitions(), array_flip($this->getFieldsToSkipFromTranslationChangesCheck($entity)));
    $field_definitions = $entity->getFieldDefinitions();

    foreach (Element::children($element) as $key) {
      if (!isset($element[$key]['#type'])) {
        $this->entityFormSharedElements($element[$key], $form_state, $form);
      }
      else {
        // Ignore non-widget form elements.
        if (isset($ignored_types[$element[$key]['#type']])) {
          continue;
        }
        // Elements are considered to be non multilingual by default.
//        if (empty($element[$key]['#multilingual'])) {
        // If we are displaying a multilingual entity form we need to provide
        // translatability clues, otherwise the non-multilingual form elements
        // should be hidden.
//          if (!$translation_form) {
        if ($display_translatability_clue && isset($field_definitions[$key])) {
          $resolver = \Drupal::service('sites_content_overrides.site_aware_entity_resolver');
          $canonical_entity = $resolver->loadCanonicalRevision($entity->getEntityType(), $entity->id());
          $main_property = $canonical_entity->get($field_definitions[$key]->getName())->getFieldDefinition()->getFieldStorageDefinition()->getMainPropertyName();
          $field_name = $field_definitions[$key]->getName();
          if (!$resolver->isFieldOverrideable($field_name, $entity)) {
            $this->addTranslatabilityClue($element[$key], FALSE, FALSE, $field_name);

            // Hide widgets for untranslatable fields.
            if ($hide_untranslatable_fields && isset($field_definitions[$key])) {
              $element[$key]['#access'] = FALSE;
              $display_warning = TRUE;
            }
          }
          else {
            $is_overridden = $canonical_entity->{$key}->{$main_property} != $entity->{$key}->{$main_property};
            $this->addTranslatabilityClue($element[$key], TRUE, $is_overridden, $field_name);
            // Provide JS settings only for overrideable fields so the clue can update live.
            $element['#attached']['drupalSettings']['sitesGroupOverrides'][$field_name] = [
              'original_value' => $canonical_entity->get($field_name)->{$main_property} ?? '',
              'label_selector' => 'span.sites-group-overrides-clue--' . $field_name,
              'value_selector' => '[name^="'.$field_name.'"][name$="['.$main_property.']"]',
            ];
            $element['#attached']['library'][] = 'sites_content_overrides/content_override_status';
          }
        }
//          }
//          else {
//            $element[$key]['#access'] = FALSE;
//          }
      }
//      }
    }

    if ($display_warning) {
      $url = $entity->getUntranslated()->toUrl('edit-form')->toString();
      // @todo Correct message.
      $message['warning'][] = $this->t('Fields that apply to all languages are hidden to avoid conflicting changes. <a href=":url">Edit them on the original language form</a>.', [':url' => $url]);
      // Explicitly renders this warning message. This prevents repetition on
      // AJAX operations or form submission. Other messages will be rendered in
      // the default location.
      // @see \Drupal\Core\Render\Element\StatusMessages.
      $element['hidden_fields_warning_message'] = [
        '#theme' => 'status_messages',
        '#message_list' => $message,
        '#weight' => -100,
        '#status_headings' => [
          'warning' => $this->t('Warning message'),
        ],
      ];
    }

    return $element;
  }

  /**
   * Adds a clue about the form element translatability.
   *
   * If the given element does not have a #title attribute, the function is
   * recursively applied to child elements.
   *
   * @param array $element
   *   A form element array.
   */
  protected function addTranslatabilityClue(&$element, $foo = FALSE, $is_overridden = FALSE, $field_name = NULL) {
    static $fapi_title_elements;

    // Elements which can have a #title attribute according to FAPI Reference.
    if (!isset($fapi_title_elements)) {
      $fapi_title_elements = array_flip(['checkbox', 'checkboxes', 'date', 'details', 'fieldset', 'file', 'item', 'password', 'password_confirm', 'radio', 'radios', 'select', 'text_format', 'textarea', 'textfield', 'weight']);
    }

    $text = t('all sites');
    if ($foo) {
      $text = $this->t('Original value');
      if ($is_overridden) {
        $text = $this->t('Overridden');
      }
    }
    $clue_classes = ['translation-entity-current-site'];
    if (!empty($field_name)) {
      $clue_classes[] = 'sites-group-overrides-clue--' . $field_name;
    }
    $suffix = ' <span class="' . implode(' ', $clue_classes) . '">(' . $text . ')</span>';

    // Update #title attribute for all elements that are allowed to have a
    // #title attribute according to the Form API Reference. The reason for this
    // check is because some elements have a #title attribute even though it is
    // not rendered; for instance, field containers.
    if (isset($element['#type']) && isset($fapi_title_elements[$element['#type']]) && isset($element['#title'])) {
      $element['#title'] .= $suffix;
    }
    // If the current element does not have a (valid) title, try child elements.
    elseif ($children = Element::children($element)) {
      foreach ($children as $delta) {
        $this->addTranslatabilityClue($element[$delta], $foo, $is_overridden, $field_name);
      }
    }
    // If there are no children, fall back to the current #title attribute if it
    // exists.
    elseif (isset($element['#title'])) {
      $element['#title'] .= $suffix;
    }

    if (!$foo) {
      $element['#disabled'] = TRUE;
    }
  }

}

