<?php

declare(strict_types=1);

namespace Drupal\sites_content_overrides\Plugin\Derivative;

use Drupal\Component\Plugin\Derivative\DeriverBase;
use Drupal\Core\Plugin\Discovery\ContainerDeriverInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\sites\SitesServiceInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Derives local tasks (tabs) for Site Overrides per supported entity type.
 *
 * Emits per-entity-type tasks:
 * - Primary tab on canonical: "Site Overrides" → list route.
 * - Primary tab on override context: "Site Overrides" → list route (so the
 *   parent tab exists when viewing/editing a specific site override).
 * - Secondary subtabs under the parent: List (default), View, Edit.
 */
final class SiteOverridesLocalTask extends DeriverBase implements ContainerDeriverInterface {

  use StringTranslationTrait;

  public function __construct(
    private readonly SitesServiceInterface $sitesService,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, $base_plugin_id) {
    return new self(
      $container->get('sites.service'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getDerivativeDefinitions($base_plugin_definition) {
    $this->derivatives = [];

    /** @var \Drupal\Core\Entity\EntityTypeManagerInterface $etm */
    $etm = \Drupal::service('entity_type.manager');
    foreach ($etm->getDefinitions() as $entity_type_id => $entity_type) {
      if (!$entity_type instanceof EntityTypeInterface) {
        continue;
      }
      if (!$entity_type->entityClassImplements(ContentEntityInterface::class)) {
        continue;
      }
      if (!$entity_type->isRevisionable()) {
        continue;
      }
      if (!$this->sitesService->entityTypeSupportsSiteContent($entity_type)) {
        continue;
      }


      $list_route_name = 'entity.' . $entity_type_id . '.site_overrides_list';
      $canonical_base = 'entity.' . $entity_type_id . '.canonical';

      // Option B: keep a single parent tab attached to the canonical base
      // that links to the list. No override-context primaries or subtabs.
      $primary_canonical_id = $entity_type_id . '.primary';
      $this->derivatives[$primary_canonical_id] = array_replace($base_plugin_definition, [
        'route_name' => $list_route_name,
        'base_route' => $canonical_base,
        'title' => 'Site Overrides',
        'weight' => 40,
      ]);

      // Option B variant: provide List/View/Edit subtabs on the override view base,
      // but do not add extra primaries (no parent row on override pages).
      $view_route_name = 'entity.' . $entity_type_id . '.site_override_view';
      $edit_route_name = 'entity.' . $entity_type_id . '.site_override_edit';

      // Add a List tab that links back to the list page but appears in the
      // override context by sharing the same base_route.
//      $this->derivatives[$entity_type_id . '.override.list'] = array_replace($base_plugin_definition, [
//        'route_name' => $list_route_name,
//        'base_route' => $view_route_name,
//        'title' => 'List',
//        'weight' => -10,
//      ]);

      // Default on override view route.
      $this->derivatives[$entity_type_id . '.override.view'] = array_replace($base_plugin_definition, [
        'route_name' => $view_route_name,
        'base_route' => $view_route_name,
        'title' => 'View override',
        'weight' => 0,
      ]);
      // Sibling Edit on override view base.
      $this->derivatives[$entity_type_id . '.override.edit'] = array_replace($base_plugin_definition, [
        'route_name' => $edit_route_name,
        'base_route' => $view_route_name,
        'title' => 'Edit override',
        'weight' => 10,
      ]);

      // Revisions subtab under the override view base.
      $revisions_route_name = 'entity.' . $entity_type_id . '.site_override_revisions';
      $this->derivatives[$entity_type_id . '.override.revisions'] = array_replace($base_plugin_definition, [
        'route_name' => $revisions_route_name,
        'base_route' => $view_route_name,
        'title' => 'Revisions',
        'weight' => 20,
      ]);

      // Layout subtab: now provided by the site-aware Layout Builder section
      // storage plugin (SiteOverridesSectionStorage). Avoid duplicating it here.
      // If needed, we could point to our custom route:
      // $lb_view_route = 'layout_builder.site_overrides.' . $entity_type_id . '.view';
      // But rely on the plugin's local task instead.
    }

    return $this->derivatives;
  }
}
