<?php

declare(strict_types=1);

namespace Drupal\sites_content_overrides\Plugin\SectionStorage;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Cache\RefinableCacheableDependencyInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Drupal\layout_builder\Attribute\SectionStorage;
use Drupal\layout_builder\Entity\LayoutBuilderEntityViewDisplay;
use Drupal\layout_builder\OverridesSectionStorageInterface;
use Drupal\layout_builder\Plugin\SectionStorage\OverridesSectionStorage;
use Drupal\layout_builder\SectionStorage\SectionStorageLocalTaskProviderInterface;
use Drupal\layout_builder\SectionStorage\SectionStorageManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Routing\RouteCollection;

/**
 * Site-aware Layout Builder overrides storage.
 *
 * This mirrors core's OverridesSectionStorage, but mounts the UI under the
 * site-override context (adds the {site} path parameter) and ensures saving
 * results in a site-specific revision (sets the site_id field).
 */
#[SectionStorage(
  id: "site_overrides",
  weight: -30,
  context_definitions: [
    'entity' => new \Drupal\Core\Plugin\Context\ContextDefinition(
      data_type: 'entity',
      label: new \Drupal\Core\StringTranslation\TranslatableMarkup("Entity")
    ),
    'view_mode' => new \Drupal\Core\Plugin\Context\ContextDefinition(
      data_type: 'string',
      label: new \Drupal\Core\StringTranslation\TranslatableMarkup("View mode"),
      default_value: "default",
    ),
  ],
  handles_permission_check: TRUE,
)]
final class SiteOverridesSectionStorage extends OverridesSectionStorage {

  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager, EntityFieldManagerInterface $entity_field_manager, SectionStorageManagerInterface $section_storage_manager, EntityRepositoryInterface $entity_repository, AccountInterface $current_user) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $entity_type_manager, $entity_field_manager, $section_storage_manager, $entity_repository, $current_user);
  }

  /**
   * {@inheritdoc}
   */
//  public function getStorageId() {
//    // Ensure the storage ID is unique per site so tempstore/locks do not clash
//    // with canonical or other site override sessions for the same entity.
//    $entity = $this->getEntity();
//    $site_id = $this->resolveSiteId() ?: 'default';
//    return $entity->getEntityTypeId() . '.' . $entity->id() . '.site.' . $site_id;
//  }
//
  /**
   * {@inheritdoc}
   */
  public function getTempstoreKey() {
    // Base key from core includes view mode and language; extend with site id
    // so parallel editing per site does not conflict or produce false "already
    // edited" messages.
    $key = parent::getTempstoreKey();
    $key .= '.site.' . ($this->resolveSiteId() ?: 'default');
    return $key;
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager'),
      $container->get('plugin.manager.layout_builder.section_storage'),
      $container->get('entity.repository'),
      $container->get('current_user'),
    );
  }

  /**
   * Build routes for Layout Builder UI under the site override context.
   */
  public function buildRoutes(RouteCollection $collection) {
    foreach ($this->getEntityTypes() as $entity_type_id => $entity_type) {
      // If the canonical route does not exist, skip.
      if (!$collection->get("entity.$entity_type_id.canonical")) {
        continue;
      }

      $defaults = [];
      $defaults['entity_type_id'] = $entity_type_id;

      // Inherit the requirements from the canonical route as a baseline.
      $requirements = $collection->get("entity.$entity_type_id.canonical")->getRequirements();

      // Add site-override specific access requirements.
      // Reuse the same access as the site override edit route.
      $requirements["_entity_access"] = "$entity_type_id.override";
      $requirements["_custom_access"] = '\\Drupal\\sites_content_overrides\\Access\\SiteOverrideEditAccessCheck::access';

      $options = [];
      // Upcasting order.
      $options['parameters']['section_storage'] = [];
      $options['parameters'][$entity_type_id]['type'] = 'entity:' . $entity_type_id;
      $options['parameters']['site']['type'] = 'string';
      $options['_admin_route'] = FALSE;

      // Mount the LB UI under the site path.
      $template = $entity_type->getLinkTemplate('canonical') . '/site/{site}/layout';

      $this->buildLayoutRoutes($collection, $this->getPluginDefinition(), $template, $defaults, $requirements, $options, $entity_type_id, $entity_type_id);
    }
  }

  /**
   * Provide a Layout tab under canonical base when LB is active.
   */
  public function buildLocalTasks($base_plugin_definition) {
    $local_tasks = [];
    foreach ($this->getEntityTypes() as $entity_type_id => $entity_type) {
      $local_tasks["layout_builder.site_overrides.$entity_type_id.view"] = $base_plugin_definition + [
        'route_name' => "layout_builder.site_overrides.$entity_type_id.view",
        'weight' => 30,
        'title' => t('Layout'),
        // Attach to the site-override view base route so it appears as a subtab
        // when viewing a site override.
        'base_route' => "entity.$entity_type_id.site_override_view",
        'cache_contexts' => ['layout_builder_is_active:' . $entity_type_id],
      ];
    }
    return $local_tasks;
  }

  /**
   * Ensure redirects go back to the site override view.
   */
  public function getRedirectUrl() {
    $entity = $this->getEntity();
    $route_parameters[$entity->getEntityTypeId()] = $entity->id();
    $route_parameters['site'] = $this->resolveSiteId() ?: 'default';
    return Url::fromRoute('entity.' . $entity->getEntityTypeId() . '.site_override_view', $route_parameters);
  }

  /**
   * Build URLs for this storage type (ensures router name matches our ID).
   */
  public function getLayoutBuilderUrl($rel = 'view') {
    $entity = $this->getEntity();
    $route_parameters[$entity->getEntityTypeId()] = $entity->id();
    $route_parameters['site'] = $this->resolveSiteId() ?: 'default';
    return Url::fromRoute("layout_builder.{$this->getStorageType()}.{$entity->getEntityTypeId()}.$rel", $route_parameters);
  }

  /**
   * Ensure saving creates a site-specific revision.
   */
  public function save() {
    // Use the entity being edited by Layout Builder to preserve the
    // original changed timestamp captured at form build time. Reloading a
    // fresh entity here can trigger the EntityChangedConstraint.
    $entity = $this->getEntity();

    if ($entity instanceof FieldableEntityInterface) {
      // Ensure a new revision is created for LB changes.
      if (method_exists($entity, 'setNewRevision')) {
        $entity->setNewRevision(TRUE);
      }
      // Bump the changed time to now so the EntityChanged constraint does not
      // think we are saving an out-of-date object (the validator compares the
      // stored changed time to the entity's current changed time before save).
      $request_time = \Drupal::time()->getRequestTime();
      if (method_exists($entity, 'setChangedTime')) {
        $entity->setChangedTime($request_time);
      }
      elseif ($entity->hasField('changed')) {
        $entity->set('changed', $request_time);
      }
      // Limit revision to active translation and mark as not translation-affecting.
      $langcode = $entity->language()->getId();
      $translation = $entity->hasTranslation($langcode) ? $entity->getTranslation($langcode) : $entity;
      if (method_exists($translation, 'setRevisionTranslationAffected')) {
        $translation->setRevisionTranslationAffected(FALSE);
      }
      // Persist the site context on the revision if available.
      if ($entity->hasField('site_id')) {
        $site_id = $this->resolveSiteId();
        if (!empty($site_id)) {
          $entity->set('site_id', $site_id);
        }
      }
    }

    return $entity->save();
  }

  /**
   * Resolves the current site ID for LB operations reliably.
   *
   * Tries, in order: current route match param, current request attribute,
   * entity's site_id field (if present), and the current_site service.
   */
  protected function resolveSiteId(): ?string {
    // 1) Route match parameter if available.
    $site = \Drupal::routeMatch()->getParameter('site');
    if (is_string($site) && $site !== '') {
      return $site;
    }
    // 2) Request attribute (for some LB sub-requests/actions).
    $request = \Drupal::requestStack()->getCurrentRequest();
    if ($request) {
      $attr = $request->attributes->get('site');
      if (is_string($attr) && $attr !== '') {
        return $attr;
      }
    }
    // 3) Entity field if already set.
    $entity = $this->getEntity();
    if ($entity instanceof FieldableEntityInterface && $entity->hasField('site_id')) {
      $val = (string) ($entity->get('site_id')->value ?? '');
      if ($val !== '') {
        return $val;
      }
    }
    // 4) Fallback to current_site service.
    try {
      /** @var \Drupal\sites\SiteProxyInterface $current_site */
      $current_site = \Drupal::service('current_site');
      $sid = $current_site->getSite()->id();
      if (is_string($sid) && $sid !== '') {
        return $sid;
      }
    }
    catch (\Throwable $e) {
      // Ignore and return null.
    }
    return NULL;
  }

  /**
   * Add site cache context applicability.
   */
  public function isApplicable(RefinableCacheableDependencyInterface $cacheability) {
    $is_applicable = parent::isApplicable($cacheability);
    if ($is_applicable) {
      $cacheability->addCacheContexts(['site']);
    }
    return $is_applicable;
  }

  /**
   * Restrict to entity types that support LB and have canonical route.
   *
   * Copied from core with same conditions.
   */
  protected function getEntityTypes() {
    return array_filter($this->entityTypeManager->getDefinitions(), function (EntityTypeInterface $entity_type) {
      return $entity_type->entityClassImplements(FieldableEntityInterface::class)
        && $entity_type->hasHandlerClass('form', 'layout_builder')
        && $entity_type->hasViewBuilderClass()
        && $entity_type->hasLinkTemplate('canonical');
    });
  }

  /**
   * Access: combine core LB access with site-override edit access.
   */
  public function access($operation, ?AccountInterface $account = NULL, $return_as_object = FALSE) {
    $result = parent::access($operation, $account, TRUE);
    $entity = $this->getEntity();
    $extra = AccessResult::neutral();
    if ($entity) {
      $extra = $entity->access('override', $account, TRUE);
    }
    $combined = $result->andIf($extra);
    return $return_as_object ? $combined : $combined->isAllowed();
  }
}
