<?php

declare(strict_types=1);

namespace Drupal\sites_content_overrides\Plugin\Validation\Constraint;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\Plugin\Validation\Constraint\EntityChangedConstraintValidator;
use Drupal\Core\Entity\RevisionableStorageInterface;
use Symfony\Component\Validator\Constraint;

/**
 * Validates the SiteAwareEntityChanged constraint.
 *
 * This validator extends the core EntityChangedConstraintValidator to support
 * site-specific revisions. It only compares changed timestamps within the same
 * site context, allowing concurrent editing across different sites while
 * maintaining protection against concurrent edits within the same site.
 */
class SiteAwareEntityChangedConstraintValidator extends EntityChangedConstraintValidator {

  /**
   * {@inheritdoc}
   */
  public function validate($entity, Constraint $constraint): void {
    if (!isset($entity)) {
      return;
    }

    /** @var \Drupal\Core\Entity\EntityInterface $entity */
    if ($entity->isNew()) {
      return;
    }

    // Check if this entity type supports site overrides via the site_id field.
    if (!($entity instanceof ContentEntityInterface) || !$entity->hasField('site_id')) {
      // Fall back to parent validation for entities without site support.
      parent::validate($entity, $constraint);
      return;
    }

    // Get the site_id from the entity being saved.
    $site_id = $entity->get('site_id')->value;

    // Load the appropriate saved entity for comparison based on site context.
    $saved_entity = $this->loadSavedEntityForComparison($entity, $site_id);

    // If no saved entity exists (e.g., first time creating a site override),
    // there's nothing to compare against - allow the save.
    if (!$saved_entity) {
      return;
    }

    // Ensure that all entity translations are the same as or newer than their
    // current version in storage to avoid reverting other changes.
    $common_translation_languages = array_intersect_key(
      $entity->getTranslationLanguages(),
      $saved_entity->getTranslationLanguages()
    );

    foreach (array_keys($common_translation_languages) as $langcode) {
      // Compare changed timestamps for each translation individually.
      if ($saved_entity->getTranslation($langcode)->getChangedTime() > $entity->getTranslation($langcode)->getChangedTime()) {
        $this->context->addViolation($constraint->message);
        break;
      }
    }
  }

  /**
   * Loads the saved entity appropriate for comparison based on site context.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity being saved.
   * @param string|null $site_id
   *   The site ID from the entity being saved (NULL or empty for canonical).
   *
   * @return \Drupal\Core\Entity\ContentEntityInterface|null
   *   The saved entity to compare against, or NULL if none exists.
   */
  protected function loadSavedEntityForComparison(ContentEntityInterface $entity, ?string $site_id): ?ContentEntityInterface {
    $entity_type_id = $entity->getEntityTypeId();
    $entity_id = $entity->id();

    $entity_type_manager = \Drupal::entityTypeManager();
    $storage = $entity_type_manager->getStorage($entity_type_id);

    if (!$storage instanceof RevisionableStorageInterface) {
      // Not revisionable, use standard load.
      return $storage->loadUnchanged($entity_id);
    }

    $entity_type = $entity->getEntityType();
    $id_key = $entity_type->getKey('id');
    $rev_key = $entity_type->getKey('revision');

    // Build query to find the latest revision matching the site context.
    $query = $storage->getQuery()->allRevisions();
    $query->condition($id_key, $entity_id);

    // If site_id is empty or NULL, we're editing the canonical revision.
    // Otherwise, we're editing a site-specific override.
    if (empty($site_id)) {
      // For canonical revisions, find the latest revision without a site_id.
      $group = $query->orConditionGroup()
        ->condition('site_id', NULL, 'IS NULL')
        ->condition('site_id', '');
      $query->condition($group);
    }
    else {
      // For site overrides, find the latest revision with matching site_id.
      $query->condition('site_id', $site_id);
    }

    $query->sort($rev_key, 'DESC');
    $query->range(0, 1);
    $query->accessCheck(FALSE);

    $ids = $query->execute();
    if (empty($ids)) {
      return NULL;
    }

    $latest_rev_id = (int) array_key_first($ids);
    $saved_entity = $storage->loadRevision($latest_rev_id);

    return $saved_entity instanceof ContentEntityInterface ? $saved_entity : NULL;
  }

}
