<?php

declare(strict_types=1);

namespace Drupal\sites_content_overrides\Routing;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Routing\RouteSubscriberBase;
use Drupal\sites_content_overrides\Form\BaseFieldConfigForm;
use Symfony\Component\Routing\Route;

/**
 * Adds explicit site-override edit routes for supported entity types.
 */
final class SiteOverrideRouteSubscriber extends RouteSubscriberBase {

  /**
   * {@inheritdoc}
   */
  protected function alterRoutes(\Symfony\Component\Routing\RouteCollection $collection): void {
    // Build routes for each content entity type that supports site content.
    /** @var \Drupal\sites\SitesServiceInterface $sites_service */
    $sites_service = \Drupal::service('sites.service');
    /** @var \Drupal\Core\Entity\EntityTypeManagerInterface $etm */
    $etm = \Drupal::service('entity_type.manager');

    foreach ($etm->getDefinitions() as $entity_type_id => $entity_type) {
      if (!$entity_type instanceof EntityTypeInterface) {
        continue;
      }
      if (!$entity_type->entityClassImplements(ContentEntityInterface::class)) {
        continue;
      }
      if (!$entity_type->isRevisionable()) {
        continue;
      }
      if (!$sites_service->entityTypeSupportsSiteContent($entity_type)) {
        continue;
      }

      // 1) Add the explicit site-override edit route (already used by forms).
      $edit_route_name = 'entity.' . $entity_type_id . '.site_override_edit';
      if (!$collection->get($edit_route_name)) {
        $edit_path = '/' . $entity_type_id . '/{' . $entity_type_id . '}/site/{site}/edit';
        $edit_route = new Route($edit_path);
        $edit_route->setDefault('_entity_form', $entity_type_id . '.site_override');
//        $edit_route->setDefault('_title_callback', '\\Drupal\\sites_content_overrides\\Controller\\SiteOverrideTitleCallback::title');
        $edit_route->setRequirement('_entity_access', $entity_type_id . '.override');
        $edit_route->setRequirement('_custom_access', '\\Drupal\\sites_content_overrides\\Access\\SiteOverrideEditAccessCheck::access');
        $edit_route->setOption('parameters', [
          $entity_type_id => [
            'type' => 'entity:' . $entity_type_id,
          ],
          'site' => [
            'type' => 'string',
          ],
        ]);
        $edit_route->setOption('_admin_route', TRUE);
        $collection->add($edit_route_name, $edit_route);
      }

      // 2) Add the per-entity list route used by the "Site Overrides" tab.
      $list_route_name = 'entity.' . $entity_type_id . '.site_overrides_list';
      if (!$collection->get($list_route_name)) {
        $list_path = '/' . $entity_type_id . '/{' . $entity_type_id . '}/site-overrides';
        $list_route = new Route($list_path);
        $list_route->setDefault('_controller', '\\Drupal\\sites_content_overrides\\Controller\\SiteOverridesListController::listing');
        $list_route->setDefault('_title', 'Site Overrides');
        // @todo Fix access, currently open for everyone.
        $list_route->setRequirement('_entity_access', $entity_type_id . '.override');
        $list_route->setRequirement('_custom_access', '\\Drupal\\sites_content_overrides\\Access\\SiteOverrideEditAccessCheck::access');
        $list_route->setOption('parameters', [
          $entity_type_id => [
            'type' => 'entity:' . $entity_type_id,
          ],
        ]);
        $list_route->setOption('_admin_route', TRUE);
        $collection->add($list_route_name, $list_route);
      }

      // 3) Add generic site-override view route (entity-type agnostic), used for subtabs.
      $generic_view_name = 'entity.' . $entity_type_id . '.site_override_view';
      if (!$collection->get($generic_view_name)) {
        $view_route = new Route('/' . $entity_type_id . '/{' . $entity_type_id . '}/site/{site}');
        $view_route->setDefault('_controller', '\\Drupal\\sites_content_overrides\\Controller\\SiteOverrideViewController::view');
        $view_route->setDefault('_title_callback', '\\Drupal\\sites_content_overrides\\Controller\\SiteOverrideTitleCallback::viewTitle');
        $view_route->setRequirement('_custom_access', '\\Drupal\\sites_content_overrides\\Access\\SiteOverrideEditAccessCheck::access');
        $view_route->setOption('parameters', [
          $entity_type_id => [
            'type' => 'entity:' . $entity_type_id,
          ],
          'site' => ['type' => 'string'],
        ]);
//        $view_route->setOption('_admin_route', TRUE);
        $collection->add($generic_view_name, $view_route);
      }

      // 4) Add per-entity revert confirmation route.
      $revert_route_name = 'entity.' . $entity_type_id . '.site_override_revert';
      if (!$collection->get($revert_route_name)) {
        $revert_path = '/' . $entity_type_id . '/{' . $entity_type_id . '}/site/{site}/revert';
        $revert_route = new Route($revert_path);
        $revert_route->setDefault('_form', '\\Drupal\\sites_content_overrides\\Form\\RevertOverrideConfirmForm');
        $revert_route->setDefault('_title', 'Revert to canonical');
        $revert_route->setRequirement('_entity_access', $entity_type_id . '.override');
        $revert_route->setRequirement('_custom_access', '\\Drupal\\sites_content_overrides\\Access\\SiteOverrideEditAccessCheck::access');
        $revert_route->setOption('parameters', [
          $entity_type_id => [
            'type' => 'entity:' . $entity_type_id,
          ],
          'site' => [
            'type' => 'string',
          ],
        ]);
        $revert_route->setOption('_admin_route', TRUE);
        $collection->add($revert_route_name, $revert_route);
      }

      // 5) Add per-entity site-override revisions overview route.
      $revisions_route_name = 'entity.' . $entity_type_id . '.site_override_revisions';
      if (!$collection->get($revisions_route_name)) {
        $revisions_path = '/' . $entity_type_id . '/{' . $entity_type_id . '}/site/{site}/revisions';
        $revisions_route = new Route($revisions_path);
        $revisions_route->setDefault('_controller', '\\Drupal\\sites_content_overrides\\Controller\\SiteOverrideRevisionsController::listing');
        $revisions_route->setDefault('_title', 'Override revisions');
        $revisions_route->setRequirement('_entity_access', $entity_type_id . '.override');
        $revisions_route->setRequirement('_custom_access', '\\Drupal\\sites_content_overrides\\Access\\SiteOverrideEditAccessCheck::access');
        $revisions_route->setOption('parameters', [
          $entity_type_id => [
            'type' => 'entity:' . $entity_type_id,
          ],
          'site' => ['type' => 'string'],
        ]);
        $revisions_route->setOption('_admin_route', TRUE);
        $collection->add($revisions_route_name, $revisions_route);
      }

      // 6) Add per-entity restore-to-revision confirm route.
      $restore_route_name = 'entity.' . $entity_type_id . '.site_override_restore';
      if (!$collection->get($restore_route_name)) {
        $restore_path = '/' . $entity_type_id . '/{' . $entity_type_id . '}/site/{site}/restore/{revision}';
        $restore_route = new Route($restore_path);
        $restore_route->setDefault('_form', '\\Drupal\\sites_content_overrides\\Form\\RestoreOverrideRevisionConfirmForm');
        $restore_route->setDefault('_title', 'Restore override revision');
        $restore_route->setRequirement('_entity_access', $entity_type_id . '.override');
        $restore_route->setRequirement('_custom_access', '\\Drupal\\sites_content_overrides\\Access\\SiteOverrideEditAccessCheck::access');
        $restore_route->setOption('parameters', [
          $entity_type_id => [
            'type' => 'entity:' . $entity_type_id,
          ],
          'site' => ['type' => 'string'],
          'revision' => ['type' => 'integer'],
        ]);
        $restore_route->setOption('_admin_route', TRUE);
        $collection->add($restore_route_name, $restore_route);
      }
    }
  }
}
