<?php

declare(strict_types=1);

namespace Drupal\sites_content_overrides\Service;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\sites_content_overrides\SiteAwareEntityResolverInterface;
use Drupal\sites\SitesServiceInterface;

/**
 * Computes how many sites have overrides for a given entity.
 */
final class SiteOverrideCounter implements SiteOverrideCounterInterface {

  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly SitesServiceInterface $sitesService,
    private readonly SiteAwareEntityResolverInterface $resolver,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function countOverridesForEntity(ContentEntityInterface $entity, ?AccountInterface $account = NULL): int {
    $account = $account ?: \Drupal::currentUser();

    // Determine candidate sites for the given user and entity.
    $candidates = $this->sitesService->getSiteCandiatesForEntityAndUser($entity, $account) ?? [];
    if (empty($candidates)) {
      return 0;
    }

    $count = 0;
    foreach ($candidates as $site_id => $site) {
      if ($this->hasOverrideForSite($entity, (string) $site_id)) {
        $count++;
      }
    }
    return $count;
  }

  /**
   * {@inheritdoc}
   */
  public function hasOverrideForSite(ContentEntityInterface $entity, string $site_id): bool {
    $storage = $this->entityTypeManager->getStorage($entity->getEntityTypeId());
    if (!$storage instanceof \Drupal\Core\Entity\RevisionableStorageInterface) {
      return FALSE;
    }

    $site = \Drupal::service('plugin.manager.site')->getSite($site_id);
    $entity_type = $entity->getEntityType();
    $id_key = $entity_type->getKey('id');
    $rev_key = $entity_type->getKey('revision');

    $query = $storage->getQuery()->allRevisions();
    $query->condition($id_key, $entity->id());
    $query->condition('site_id', $site_id);
    $query->range(0, 1);
    $query->accessCheck(TRUE);
    $query->sort($rev_key, 'DESC');
    $ids = $query->execute();

    if (empty($ids)) {
      return FALSE;
    }

    // Get the latest revision id from the result set.
    $rev_id = function_exists('array_key_first') ? array_key_first($ids) : key($ids);
    if (!$rev_id) {
      $rev_id = reset($ids) ?: NULL;
    }
    if (!$rev_id) {
      return FALSE;
    }

    $override = $storage->loadRevision($rev_id);
    if (!$override instanceof ContentEntityInterface) {
      return FALSE;
    }

    // Consider as an override only if there is at least one overrideable field
    // with a non-empty value.
    foreach ($override->getFieldDefinitions() as $field_name => $definition) {
      if (in_array($field_name, [$entity_type->getKey('revision'), $entity_type->getKey('id'), 'site_id'], TRUE)) {
        continue;
      }
      if (!$this->resolver->isFieldOverrideable($field_name, $override, $site)) {
        continue;
      }
      if ($override->hasField($field_name) && !$override->get($field_name)->isEmpty()) {
        return TRUE;
      }
    }

    return FALSE;
  }
}
