<?php

declare(strict_types=1);

namespace Drupal\sites_content_overrides\Service;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Service interface to compute override counts/status for entities.
 */
interface SiteOverrideCounterInterface {

  /**
   * Counts how many sites have an effective override for the given entity.
   *
   * A site is considered "overridden" when the latest site-specific revision
   * for that site contains at least one overrideable field with a non-empty
   * value. This matches the logic used by the listing controller.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to check. Must be revisionable and site-aware in your module.
   * @param \Drupal\Core\Session\AccountInterface|null $account
   *   Optional account to filter site candidates for. Defaults to current user.
   *
   * @return int
   *   Number of sites that currently have an effective override.
   */
  public function countOverridesForEntity(ContentEntityInterface $entity, ?AccountInterface $account = NULL): int;

  /**
   * Checks if the given entity has an effective override for a specific site.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to check.
   * @param string $site_id
   *   The site ID.
   *
   * @return bool
   *   TRUE if an override exists per the module semantics, FALSE otherwise.
   */
  public function hasOverrideForSite(ContentEntityInterface $entity, string $site_id): bool;
}
