<?php

declare(strict_types=1);

namespace Drupal\sites_content_overrides;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Entity\RevisionableStorageInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\sites\SiteInterface;
use Drupal\sites\SiteProxyInterface;
use Drupal\sites\SitesServiceInterface;
use Drupal\user\UserInterface;

/**
 * Default implementation of the site-aware entity resolver.
 */
final class SiteAwareEntityResolver implements SiteAwareEntityResolverInterface {

  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly SiteProxyInterface $currentSite,
    private readonly SitesServiceInterface $sitesService,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function loadEffectiveById(string $entity_type_id, int|string $entity_id, ?SiteInterface $site = NULL): ?EntityInterface {
    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id, false);
    if (!$entity_type) {
      return NULL;
    }

    $storage = $this->entityTypeManager->getStorage($entity_type_id);
    if (!$storage instanceof RevisionableStorageInterface) {
      // Not a revisionable entity, load the regular entity.
      return $storage->load($entity_id);
    }

    // If sites are not supported for this entity type, return latest revision.
    if (!$this->sitesService->entityTypeSupportsSiteContent($entity_type)) {
      $latest_id = $storage->getLatestRevisionId($entity_id);
      return $latest_id ? $storage->loadRevision($latest_id) : $storage->load($entity_id);
    }

    // Resolve current site.
    $site = $site ?: $this->currentSite->getSite();
    $site_id = $site?->id();
    if ($site_id == 'default' && \Drupal::routeMatch()->getParameter('site')) {
      $site_id = \Drupal::routeMatch()->getParameter('site');
    }

    // Load canonical and site override revisions.
    $canonical = $this->loadCanonicalRevision($entity_type, $entity_id);
    if (!$canonical) {
      // As a fallback, return latest revision (should be rare if site field is used properly).
      $latest_id = $storage->getLatestRevisionId($entity_id);
      return $latest_id ? $storage->loadRevision($latest_id) : $storage->load($entity_id);
    }

    // If no site in context, return canonical.
    if ($site_id === NULL || $site_id === '') {
      return $canonical;
    }

    $override = $this->loadSiteOverrideRevision($entity_type, $entity_id, $site_id);
    if (!$override) {
      return $canonical;
    }

    // Compose effective entity: clone canonical and overlay overrideable fields.
    $effective = clone $canonical;

    // Iterate all fields defined on the entity and copy overrideable values.
    foreach ($effective->getFieldDefinitions() as $field_name => $definition) {
      // Skip internal base fields that shouldn't be merged like site_id and revision IDs.
      if (in_array($field_name, [$entity_type->getKey('revision'), $entity_type->getKey('id'), 'site_id'], true)) {
        continue;
      }

//      $overrideable = (bool) $definition->getThirdPartySetting('sites_content_overrides', 'overrideable', FALSE);
      $overrideable = $this->isFieldOverrideable($field_name, $effective, $site);
      // Always allow to override the changed filed to keep changed time up to date.
      if (!$overrideable && $field_name != 'changed') {
        // Always keep canonical value.
        continue;
      }

      // If override has a non-empty value for this field, copy it over.
      if ($override->hasField($field_name) && !$override->get($field_name)->isEmpty()) {
        $effective->set($field_name, $override->get($field_name)->getValue());
      }
    }

    return $effective;
  }

  /**
   * Checks if a field is overrideable.
   *
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition.
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity.
   *
   * @return bool
   *   TRUE if the field is overrideable.
   */
  public function isFieldOverrideable($field_name, $entity, $site = NULL): bool {
    if (!$site) {
      $site = $this->currentSite->getSite();
      if ($site->id() == 'default' && \Drupal::routeMatch()->getParameter('site')) {
        $site = \Drupal::service('plugin.manager.site')->getSite(\Drupal::routeMatch()->getParameter('site'));
      }
    }

    $supported_entity_type_ids = sites_supported_entity_type_ids();
    if (in_array($entity->getEntityTypeId(), $supported_entity_type_ids)) {
      $field_names = $site->getSetting('content_overrides_field_names__' . $entity->getEntityTypeId()) ?? '';
      $field_names = explode(',', $field_names);
      if (in_array($field_name, $field_names)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Loads the latest canonical (global) revision for an entity.
   */
  public function loadCanonicalRevision(EntityTypeInterface $entity_type, int|string $entity_id): ?ContentEntityInterface {
    $storage = $this->entityTypeManager->getStorage($entity_type->id());
    if (!$storage instanceof RevisionableStorageInterface) {
      return $storage->load($entity_id);
    }

    $id_key = $entity_type->getKey('id');
    $rev_key = $entity_type->getKey('revision');

    $query = $storage->getQuery()->allRevisions();
    $query->condition($id_key, $entity_id);
    $group = $query->orConditionGroup()
      ->condition('site_id', NULL, 'IS NULL')
      ->condition('site_id', '');
    $query->condition($group);
    $query->sort($rev_key, 'DESC');
    $query->range(0, 1);
    $query->accessCheck(FALSE);
    $ids = $query->execute();
    if (!empty($ids)) {
      $latest_rev_id = (int) array_key_first($ids);
      $entity = $storage->loadRevision($latest_rev_id);
      return $entity instanceof ContentEntityInterface ? $entity : NULL;
    }

    // Fallback to latest revision.
    $latest_id = $storage->getLatestRevisionId($entity_id);
    $entity = $latest_id ? $storage->loadRevision($latest_id) : $storage->load($entity_id);
    return $entity instanceof ContentEntityInterface ? $entity : NULL;
  }

  /**
   * Loads the site-specific revision if it exists.
   */
  private function loadSiteOverrideRevision(EntityTypeInterface $entity_type, int|string $entity_id, string $site_id): ?ContentEntityInterface {
    $storage = $this->entityTypeManager->getStorage($entity_type->id());

    $id_key = $entity_type->getKey('id');
    // Query for the latest revision with matching site_id
    $query = $storage->getQuery()
      ->condition($id_key, $entity_id)
      ->condition('site_id', $site_id)
      ->accessCheck(FALSE)
      ->allRevisions()
      ->sort($entity_type->getKey('revision'), 'DESC')
      ->range(0, 1);

    $result = $query->execute();

    if ($result) {
      $revision_id = key($result);
      return $storage->loadRevision($revision_id);
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function loadEffective(EntityInterface $entity, ?SiteInterface $site = NULL): ?EntityInterface {
    // Prefer to work by ID to leverage revision queries and composition.
    if (!$entity instanceof ContentEntityInterface) {
      return $entity;
    }
    if (!$entity instanceof RevisionableInterface) {
      return $entity;
    }
    if ($entity instanceof UserInterface) {
      return $entity;
    }
    $entity_type_id = $entity->getEntityTypeId();
    $id = $entity->id();
    if ($id === NULL) {
      // Unsaved entity; nothing to resolve.
      return $entity;
    }
    return $this->loadEffectiveById($entity_type_id, $id, $site);
  }

}
