<?php

declare(strict_types=1);

namespace Drupal\sites_content_overrides;

use Drupal\Core\Entity\EntityInterface;
use Drupal\sites\SiteInterface;

/**
 * Resolves the effective entity for a given site context.
 *
 * The resolver composes a runtime entity that represents the "effective"
 * values for the current site:
 * - Base = latest canonical/global revision (where site_id is NULL/empty).
 * - If a site-specific override revision exists, fields marked as
 *   overrideable (via FieldConfig third-party setting
 *   sites_content_overrides:overrideable) are copied from the override
 *   revision when non-empty; all other fields keep the canonical value.
 *
 * For entity types that do not support site content, the resolver falls back
 * to returning the latest stored revision (or the loaded entity if not
 * revisionable).
 */
interface SiteAwareEntityResolverInterface {

  /**
   * Loads the effective entity by entity type and ID.
   *
   * Note: The returned entity may be a composed, in-memory object that does
   * not correspond to a single stored revision.
   *
   * @param string $entity_type_id
   *   The entity type ID (e.g. 'node').
   * @param int|string $entity_id
   *   The entity ID.
   * @param \Drupal\sites\SiteInterface|null $site
   *   Optional explicit site. When NULL, the current site will be used.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The effective entity, or NULL if not found.
   */
  public function loadEffectiveById(string $entity_type_id, int|string $entity_id, ?SiteInterface $site = NULL): ?EntityInterface;

  /**
   * Loads the effective entity for an existing entity object.
   *
   * Note: The returned entity may be a composed, in-memory object that does
   * not correspond to a single stored revision.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity whose effective values should be resolved.
   * @param \Drupal\sites\SiteInterface|null $site
   *   Optional explicit site. When NULL, the current site will be used.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The effective entity, or NULL if not found.
   */
  public function loadEffective(EntityInterface $entity, ?SiteInterface $site = NULL): ?EntityInterface;

}
