<?php

declare(strict_types=1);

namespace Drupal\Tests\sites_migrator\Kernel;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\group\Entity\Group;
use Drupal\group\Entity\GroupType;
use Drupal\group_content_menu\Entity\GroupContentMenuType;
use Drupal\menu_link_content\Entity\MenuLinkContent;
use Drupal\sites_migrator\Menu\MenuConfigMigrator;
use Drupal\sites_migrator\Menu\MenuContentMigrator;
use Drupal\system\Entity\Menu;
use Drupal\Tests\group\Kernel\GroupKernelTestBase;
use Drupal\user\Entity\User;

/**
 * @group GroupKernelTestBase
 */
class GroupContentMenuBundleTest extends GroupKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'field',
    'group_content_menu',
    'group',
    'flexible_permissions',
    'workspaces',
    'path_alias',
    'sites',
    'sites_group',
    'sites_migrator',
    'menu_link_content',
    'link',
  ];

  private MenuConfigMigrator $menuConfigMigrator;
  private MenuContentMigrator $menuContentMigrator;

  private EntityStorageInterface $groupContentMenuTypeStorage;

  private const SOURCE_TEST_MENU_ID = 'test_menu';
  private const DESTINATION_BUNDLE = 'test_menu_destination';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('user');
    $this->installEntitySchema('group');
    $this->installEntitySchema('group_relationship');
    $this->installEntitySchema('group_config_wrapper');
    $this->installEntitySchema('group_content_menu');
    $this->installEntitySchema('menu_link_content');
    $this->installSchema('system', ['sequences']);
    $this->installConfig(['group', 'node', 'sites_group']);
    $this->menuConfigMigrator = $this->container->get('sites_migrator.menu_config_migrator');
    $this->menuContentMigrator = $this->container->get('sites_migrator.menu_content_migrator');
    $this->groupContentMenuTypeStorage = $this->container->get('entity_type.manager')->getStorage('group_content_menu_type');

    /**
     * @var \Drupal\sites_migrator\MenuConfigMigrator $menuConfigMigrator
     */
    $menuConfigMigrator = $this->menuConfigMigrator;

    $mainMenu = Menu::create([
      'id' => self::SOURCE_TEST_MENU_ID,
      'label' => 'Test menu',
    ]);
    $mainMenu->save();

    $menuConfigMigrator->migrateMenuTypeConfig(self::SOURCE_TEST_MENU_ID);
  }

  public function testMigrateMenuConfig(): void {
    $groupContentMenuType = $this->groupContentMenuTypeStorage->load('group_' . self::SOURCE_TEST_MENU_ID);
    $this->assertInstanceOf(GroupContentMenuType::class, $groupContentMenuType);
  }

  public function testMigrateMenuLinkContentException(): void {
    $this->expectException(\Exception::class);
    $this->expectExceptionMessage('The group is not a site.');
    $this->prepareGroup();

    $options = [
      'menu_name' => self::SOURCE_TEST_MENU_ID,
      'bundle' => 'menu_link_content',
      'link' => [['uri' => 'internal:/']],
      'title' => 'Link test',
    ];
    $link = MenuLinkContent::create($options);
    $link->save();

    /**
     * @var \Drupal\sites_migrator\MenuContentMigrator $menuContentMigrator
     */
    $menuContentMigrator = $this->menuContentMigrator;
    $menuContentMigrator->migrateMenuItems(
      self::SOURCE_TEST_MENU_ID,
      1,
      self::DESTINATION_BUNDLE,
      'Example menu'
    );

    $groupContentMenuStorage = $this->entityTypeManager
      ->getStorage('group_content_menu');
    $results = $groupContentMenuStorage
      ->loadByProperties([
        'bundle' => self::DESTINATION_BUNDLE,
      ]);

    /**
     * @var \Drupal\group_content_menu\Entity\GroupContentMenu $result
     */
    $result = current($results);

    $this->assertEquals(
      self::DESTINATION_BUNDLE,
      $result->bundle->getString()
    );

    $this->assertEquals(
      'group_content_menu',
      $result->getEntityType()->id()
    );

    $this->assertCount(1, $results);
  }

  /**
   * @return void
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  private function prepareGroup(): void {
    $group_type = GroupType::create([
      'id' => 'test_group',
      'label' => 'Test Group',
      'description' => 'Test Group Type',
    ]);
    $group_type->save();

    $user = User::create(['name' => $this->randomString()]);
    $user->save();
    $group = Group::create([
      'type' => 'test_group',
      'label' => 'Test Group',
      'uid' => $user->id(),
    ]);
    $group->save();
  }

}
