<?php

declare(strict_types=1);

namespace Drupal\sites_migrator\Menu;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Menu\MenuLinkTreeElement;
use Drupal\Core\Menu\MenuLinkTreeInterface;
use Drupal\Core\Menu\MenuTreeParameters;
use Drupal\group\Entity\GroupInterface;
use Drupal\sites_migrator\ContentInheritService;
use Psr\Log\LoggerInterface;
use Symfony\Component\Console\Helper\ProgressBar;
use Symfony\Component\Console\Output\ConsoleOutput;

/**
 * Functionality for migrating menu content into the sites group module context.
 */
class MenuContentMigrator implements MenuContentMigratorInterface {

  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly MenuLinkTreeInterface $menuLinkTree,
    private readonly LoggerInterface $logger,
    private readonly ContentInheritService $contentInheritService,
    private readonly MenuServiceInterface $menuService,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public function migrateMenuItems(
    string $sourceMenuName,
    int $groupId,
    string $menuBundle,
    string $menuLabel,
    ?ConsoleOutput $output = NULL,
  ): void {
    $group = $this->entityTypeManager->getStorage('group')->load($groupId);
    if (!$group instanceof GroupInterface) {
      throw new \Exception("Group with ID '$groupId' not found.");
    }

    $groupContentMenus = $this->entityTypeManager
      ->getStorage('group_content_menu')
      ->loadByProperties(['bundle' => $menuBundle]);

    if (empty($groupContentMenus)) {
      $groupContentMenuStorage = $this->entityTypeManager
        ->getStorage('group_content_menu');
      $groupContentMenu = $groupContentMenuStorage->create([
        'bundle' => $menuBundle,
        'label' => $menuLabel,
      ]);
      $groupContentMenu->save();

      $this->contentInheritService->ensureSiteContextByGroup($groupContentMenu, (string) $groupId);
    }
    else {
      $groupContentMenu = current($groupContentMenus);
      $this->contentInheritService->ensureSiteContextByGroup($groupContentMenu, (string) $groupId);
    }

    $menuTree = $this->menuLinkTree->load($sourceMenuName, new MenuTreeParameters());

    $progressBar = NULL;
    if ($output) {
      $progressBar = new ProgressBar($output, $this->menuService->countMenuItems($sourceMenuName));
      $progressBar->start();
    }

    foreach ($menuTree as $item) {
      if ($item->link) {
        try {
          $this->menuService->createMenuLinkContent(
            $groupContentMenu->bundle(),
            $groupContentMenu->id(),
            $item,
            NULL
          );
        }
        catch (\Exception $e) {
          (function (\Exception $e, MenuLinkTreeElement $item) {
            if (str_contains($e->getMessage(), 'has no corresponding route.')) {
              $this->logger->notice('Orphaned menu item detected. Skipping. See original exception: @message', ['@message' => $e->getMessage()]);
              return;
            }

            // We expect not to have a site context, if the script is executed via CLI.
            if ($e->getMessage() !== "The 'site' context is required and not present.") {
              $this->logger->error('Failed to migrate menu item "@title": @message', [
                '@title' => $item->link->getTitle(),
                '@message' => $e->getMessage(),
              ]);
            }
          })($e, $item);
        }
      }

      if ($progressBar) {
        $progressBar->advance();
      }
    }

    if ($progressBar) {
      $progressBar->finish();
      $output->writeln('');
    }
  }

}
