<?php

declare(strict_types=1);

namespace Drupal\sites_migrator;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\group\Entity\GroupInterface;
use Drupal\group\Plugin\Group\Relation\GroupRelationInterface;
use Drupal\sites_group\SitesGroupServiceInterface;
use Symfony\Component\Console\Helper\ProgressBar;
use Symfony\Component\Console\Output\ConsoleOutputInterface;

/**
 * Inherits content entities from various bundles into the group context.
 */
final class ContentInheritService implements ContentInheritServiceInterface {

  public function __construct(
    readonly private SitesGroupServiceInterface $sitesGroupService,
    readonly public EntityTypeManagerInterface $entityTypeManager,
    readonly public LoggerChannelInterface $logger,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public function inheritContent(string|null $groupId, ?ConsoleOutputInterface $output = NULL, ?array $sourceContentConditions = NULL): void {
    $groupId = $this->handleGroupId($groupId);
    $group = $this->entityTypeManager->getStorage('group')->load($groupId);
    assert($group instanceof GroupInterface, 'Group not found');
    $contentEntityIds = $this->fetchContentEntityIds($group, $sourceContentConditions);

    $total = 0;
    foreach ($contentEntityIds as $entityIds) {
      $total += count($entityIds);
    }

    $progressBar = NULL;
    if ($output) {
      $progressBar = new ProgressBar($output, $total);
      $progressBar->start();
    }

    foreach ($contentEntityIds as $entityTypeId => $entityIds) {
      $contentEntityStorage = $this->entityTypeManager->getStorage($entityTypeId);

      foreach ($entityIds as $entity_id) {
        $contentEntity = $contentEntityStorage->load($entity_id);
        try {
          $this->ensureSiteContextByGroup($contentEntity, $groupId);
        }
        catch (\Throwable $t) {
          $this->logger->error($t->getMessage());
        }

        if ($progressBar) {
          $progressBar->advance();
        }
      }
    }

    if ($progressBar) {
      $progressBar->finish();
      $output->writeln('');
    }
  }

  /**
   * {@inheritdoc}
   */
  public function ensureSiteContextByGroup(ContentEntityInterface $entity, string $groupId): void {
    $group = $this->getGroup($groupId);
    if (!$this->sitesGroupService->groupIsSite($group)) {
      throw new \Exception('The group is not a site.');
    }

    $pluginId = $this->sitesGroupService->getPluginIdByEntity($entity);
    if ($group instanceof GroupInterface && $pluginId !== NULL) {
      if ($this->sitesGroupService->isContentPluginInstalledForGroup($group, $pluginId)) {
        $relatedEntities = $group->getRelatedEntities($pluginId);

        // Check if the entity is already related to the group.
        $entityAlreadyRelated = FALSE;
        foreach ($relatedEntities as $relatedEntity) {
          if ($relatedEntity->id() === $entity->id() &&
            $relatedEntity->getEntityTypeId() === $entity->getEntityTypeId()) {
            $entityAlreadyRelated = TRUE;
            break;
          }
        }

        // Only add the relationship if the entity is not already related.
        if (!$entityAlreadyRelated) {
          $group->addRelationship(
            $entity,
            $pluginId
          );
        }
      }
    }
  }

  /**
   * @param string|null $groupId
   *   The group id or no group id, if the first group should be used.
   *
   * @return string
   * @throws \Exception
   */
  private function handleGroupId(?string $groupId): string {
    if (is_string($groupId)) {
      return $groupId;
    }

    $groups = $this->getAllGroups();

    if (count($groups) === 0) {
      throw new \Exception('No groups available to inherit content.');
    }

    if (count($groups) > 1) {
      throw new \Exception('Multiple groups found. Please specify a group ID.');
    }

    return reset($groups)->id();
  }

  private function getGroup(string $groupId): ?GroupInterface {
    $group = $this->entityTypeManager->getStorage('group')->load($groupId);
    return $group instanceof GroupInterface ? $group : NULL;
  }

  private function fetchContentEntityIds(GroupInterface $group, ?array $sourceContentConditions = NULL): array {
    $content_entity_ids = [];

    foreach ($group->getGroupType()->getInstalledPlugins() as $plugin) {
      assert($plugin instanceof GroupRelationInterface);
      $definition = $plugin->getPluginDefinition();
      if (!$entity_type_id = $definition->get('entity_type_id')) {
        continue;
      }
      $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);

      if ($entity_type->entityClassImplements(ContentEntityInterface::class)) {
        try {
          $storage = $this->entityTypeManager->getStorage($entity_type_id);
          $query = $storage->getQuery()->accessCheck(FALSE);

          // Apply source content conditions if provided.
          if ($sourceContentConditions) {
            foreach ($sourceContentConditions as $name => $value) {
              $query->condition($name, $value);
            }
          }

          $entity_ids = $query->execute();

          if (!empty($entity_ids)) {
            $content_entity_ids[$entity_type_id] = $entity_ids;
          }
        }
        catch (\Exception $e) {
          $this->logger->error('Error fetching @type entities: @message', [
            '@type' => $entity_type_id,
            '@message' => $e->getMessage(),
          ]);
        }
      }
    }

    return $content_entity_ids;
  }

  /**
   * Fetches all groups in the system.
   *
   * @return \Drupal\Core\Entity\EntityInterface[]
   *   An array of all groups, which are a site.
   */
  private function getAllGroups(): array {
    $storage = $this->entityTypeManager->getStorage('group');
    $group_ids = $storage->getQuery()->accessCheck(FALSE)->execute();
    $groupsAsSites = [];

    if (!empty($group_ids)) {
      $allGroups = $storage->loadMultiple($group_ids);
      foreach ($allGroups as $group) {
        if ($this->sitesGroupService->groupIsSite($group)) {
          $groupsAsSites[] = $group;
        }
      }
    }

    return $groupsAsSites;
  }

}
