<?php

declare(strict_types=1);

namespace Drupal\sites_migrator;

use Drupal\Core\Entity\ContentEntityInterface;
use Symfony\Component\Console\Output\ConsoleOutputInterface;

/**
 * Interface for the content inheritance service.
 *
 * Provides methods to inherit content entities from various bundles into the
 * group context, particularly for sites in any (non-core) multisite setup.
 */
interface ContentInheritServiceInterface {

  /**
   * Inherits content entities into a specified group context.
   *
   * This method finds content entities that can be related to the specified
   * group and creates those relationships. It can optionally filter the
   * content entities based on provided conditions.
   *
   * @param string|null $groupId
   *   The ID of the group to inherit content into. If NULL, will attempt to use
   *   the first available group.
   * @param \Symfony\Component\Console\Output\ConsoleOutputInterface|null $output
   *   Optional console output interface for displaying progress.
   * @param array|null $sourceContentConditions
   *   Optional array of conditions to filter the content entities. Format:
   *   ['field_name' => 'value'] or ['field_name' => ['value1', 'value2']].
   *
   * @throws \Exception
   *   Thrown when no groups are available or when multiple groups exist but no
   *   specific group ID is provided.
   */
  public function inheritContent(string|null $groupId, ?ConsoleOutputInterface $output = NULL, ?array $sourceContentConditions = NULL): void;

  /**
   * Inherits user entities into all site groups.
   *
   * This method finds user entities and creates group memberships for all site groups.
   * It can optionally filter the user entities based on provided conditions.
   *
   * @param \Symfony\Component\Console\Output\ConsoleOutputInterface|null $output
   *   Optional console output interface for displaying progress.
   * @param array|null $sourceContentConditions
   *   Optional array of conditions to filter the user entities. Format:
   *   ['field_name' => 'value'] or ['field_name' => ['value1', 'value2']].
   */
  public function inheritUsers(?ConsoleOutputInterface $output = NULL, ?array $sourceContentConditions = NULL): void;

  /**
   * Ensures a content entity is related to a specific group.
   *
   * This method creates a relationship between the provided content entity and
   * the specified group if such a relationship doesn't already exist. It verifies
   * that the group is a site and that the appropriate content plugin is installed
   * for the group.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The content entity to relate to the group.
   * @param string $groupId
   *   The ID of the group to relate the content entity to.
   * @param bool $bypassUsers
   *   Whether to bypass user entities (default: TRUE).
   *
   * @throws \Exception
   *   Thrown when the specified group is not a site or when other errors occur
   *   during the relationship creation process.
   */
  public function ensureSiteContextByGroup(ContentEntityInterface $entity, string $groupId, bool $bypassUsers = TRUE): void;
}
