<?php

declare(strict_types=1);

namespace Drupal\sites_migrator\EventSubscriber;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Site\Settings;
use Drupal\migrate\Event\MigratePostRowSaveEvent;
use Drupal\sites_migrator\ContentInheritService;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber for handling post-row save events during migration.
 */
class SitesMigrateEventSubscriber implements EventSubscriberInterface {

  /**
   * Constructs a new SitesMigrateEventSubscriber object.
   *
   * @param \Drupal\Core\Site\Settings $settings
   *   The settings service.
   * @param \Drupal\sites_migrator\ContentInheritService $contentInheritService
   *   The content inherit service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    private readonly Settings $settings,
    private readonly ContentInheritService $contentInheritService,
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    // @todo Replace this with constant, after the migrate module is installed
    // on all environments.
    return [
      'migrate.post_row_save' => 'onMigratePostRowSave',
    ];
  }

  /**
   * Reacts to a post row save event during migration.
   *
   * @param \Drupal\migrate\Event\MigratePostRowSaveEvent $event
   *   The migrate post row save event.
   *
   * @throws \Exception
   *   Throws an exception if the group name is not set or if the group cannot be determined.
   */
  public function onMigratePostRowSave(MigratePostRowSaveEvent $event): void {
    $destination_values = $event->getDestinationIdValues();
    if (count($destination_values) === 0) {
      return;
    }

    $entity_id = current($destination_values);
    $group_name = $this->settings->get('migrate_group_name');

    if (!$group_name) {
      throw new \Exception('There must be a "migrate_group_name" settings variable value set. E.g.: "example.com".');
    }

    $group_result = $this->entityTypeManager->getStorage('group')->loadByProperties(['label' => $group_name]);

    if (!$group_result) {
      throw new \Exception("Could not determine a group by label '$group_name'.");
    }

    if (count($group_result) > 1) {
      throw new \Exception("There should be only one group determined by the group label '$group_name'.");
    }

    $group = current($group_result);
    $entity_type_id = $event->getMigration()->getSourcePlugin()->current()->getSource()['entity_type'];
    $content = $this->entityTypeManager->getStorage($entity_type_id)->load($entity_id);

    $this->contentInheritService->ensureSiteContextByGroup($content, $group->id());
  }

}
