<?php

declare(strict_types=1);

namespace Drupal\sites_migrator\Menu;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Menu\MenuLinkInterface;
use Drupal\Core\Menu\MenuLinkTreeElement;
use Drupal\Core\Menu\MenuLinkTreeInterface;
use Drupal\Core\Menu\MenuTreeParameters;
use Drupal\menu_item_extras\Entity\MenuItemExtrasMenuLinkContent;
use Drupal\path_alias\AliasManagerInterface;

/**
 * Various methods for handling menus during the menu migration.
 */
class MenuService implements MenuServiceInterface {

  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly MenuLinkTreeInterface $menuLinkTree,
    private readonly AliasManagerInterface $aliasManager,
  ) {
  }

  public function createMenuLinkContent(
    $bundle,
    $groupMenuId,
    MenuLinkTreeElement $menuLinkTreeElement,
    ?MenuItemExtrasMenuLinkContent $parent = NULL,
  ): void {
    $values = [
      'bundle' => $bundle,
      'menu_name' => 'group_menu_link_content-' . $groupMenuId,
      'title' => $menuLinkTreeElement->link->getTitle(),
      'expanded' => $menuLinkTreeElement->link->isExpanded(),
      'enabled' => $menuLinkTreeElement->link->isEnabled(),
      'weight' => $menuLinkTreeElement->link->getWeight(),
      'description' => $menuLinkTreeElement->link->getDescription(),
      'view_mode' => 'default',
    ];

    $values['link'] = $this->generateUriFromRoute($menuLinkTreeElement->link);

    if ($parent) {
      $values['parent'] = 'menu_link_content:' . $parent->uuid();
    }

    $menuLinkContent = $this->entityTypeManager
      ->getStorage('menu_link_content');
    $link = $menuLinkContent
      ->create($values);
    $link->save();

    if ($menuLinkTreeElement->hasChildren) {
      foreach ($menuLinkTreeElement->subtree as $child) {
        $this->createMenuLinkContent($bundle, $groupMenuId, $child, $link);
      }
    }

    // Note: if we do have menu items with fields, then we can implement this like here:
    // \Drupal\mie_demo_base\Utility\MieDemoBaseUtility::createMieDemoBaseMenuMenuLinkContent()
  }

  public function countMenuItems(string $sourceMenuName): int {
    $menuTree = $this->menuLinkTree->load($sourceMenuName, new MenuTreeParameters());
    $count = 0;

    foreach ($menuTree as $item) {
      $count++;
      if ($item->hasChildren) {
        $count += $this->countSubtree($item->subtree);
      }
    }

    return $count;
  }

  public function countSubtree(array $subtree): int {
    $count = 0;
    foreach ($subtree as $item) {
      $count++;
      if ($item->hasChildren) {
        $count += $this->countSubtree($item->subtree);
      }
    }
    return $count;
  }

  public function generateUriFromRoute(MenuLinkInterface $menuLink): array {
    $urlObject = $menuLink->getUrlObject();

    $uri = $urlObject->toString();

    if (!$urlObject->isExternal() && strpos($uri, '://') === FALSE) {
      $uri = 'route:<nolink>';
    }

    if (!$urlObject->isExternal()) {
      if ($urlObject->isRouted()) {
        $internalPath = $urlObject->getInternalPath();
        if (strpos($internalPath, '/') !== 0) {
          $internalPath = '/' . $internalPath;
        }
        $uri = $this->aliasManager->getAliasByPath($internalPath);
      }

      // Ensure the URI is correctly formatted for internal paths.
      if (strpos($uri, '/') === 0 && strpos($uri, '://') === FALSE) {
        $uri = 'internal:' . $uri;
      }

      if ($menuLink->getUrlObject()->getRouteName() === '<nolink>' || $menuLink->getUrlObject()->getRouteName() === '') {
        $uri = 'route:<nolink>';
      }

      if (!$menuLink->getUrlObject()->isRouted()) {
        $uri = 'route:<nolink>';
      }
    }

    return [
      [
        'uri' => $uri,
        'title' => $menuLink->getTitle(),
      ],
    ];
  }

}
