<?php

declare(strict_types=1);

namespace Drupal\sites_migrator\Plugin\migrate\process;

use Drupal\Core\Database\Connection;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\migrate\Attribute\MigrateProcess;
use Drupal\migrate\Plugin\migrate\process\SubProcess;
use Drupal\migrate\MigrateExecutableInterface;
use Drupal\migrate\Row;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Sub-process for paragraphs migration. Migrates paragraphs into paragraphs.
 * Handy for e.g. migrating paragraphs from a different Drupal 10 site database
 * into a new Drupal site.
 *
 * Example usage in migration YAML file:
 *
 * field_paragraphs:
 *   plugin: paragraphs_sub_process
 *   source: field_paragraphs
 *   migration:
 *     - xyz_paragraph_media_text
 *   process:
 *     target_id: '0'
 *     target_revision_id: '1'
 */
#[MigrateProcess(
  id: "paragraphs_sub_process",
  handle_multiples: TRUE,
)]
class ParagraphsSubProcess extends SubProcess implements ContainerFactoryPluginInterface {

  /**
   * Constructs ParagraphsSubProcess object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection to be used.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected readonly Connection $database,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('database'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function transform($value, MigrateExecutableInterface $migrate_executable, Row $row, $destination_property) {
    $return = [];
    // Make a lookup for all migrations, migrating paragraphs.
    foreach ($this->configuration['migration'] ?? [] as $migration) {
      $paragraphs = $this->lookupParagraphs($migration, $value);
      $return += parent::transform($paragraphs, $migrate_executable, $row, $destination_property);
    }

    // Sort by array key, which will be the original delta of the paragraphs.
    ksort($return);
    return $return;
  }

  /**
   * Lookup paragraphs by target_id.
   *
   * This unfortunally does not work via default lookup.
   *
   * @param string $migration_id
   *   The migration id.
   * @param array $source_id_values
   *   The source ID values to look up.
   *
   * @return array
   *   The identified paragraphs.
   */
  protected function lookupParagraphs(string $migration_id, array $source_id_values): array {
    $paragraphs = [];
    foreach ($source_id_values as $delta => $source_ids) {
      if (isset($source_ids['target_id'])) {
        $query = $this->database
          ->select("migrate_map_$migration_id", 'mm')
          ->fields('mm', ['sourceid1', 'destid1', 'destid2']);
        $query->condition('mm.sourceid1', $source_ids['target_id']);
        // $query->condition('mm.sourceid2', $source_ids['target_revision_id']);

        $results = $query->execute()->fetchAll();
        foreach ($results as $result) {
          $paragraphs[$delta] = [$result->destid1, $result->destid2];
        }
      }
    }

    return $paragraphs;
  }

}
