<?php

declare(strict_types=1);

namespace Drupal\sites_migrator\Menu;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\group_content_menu\Entity\GroupContentMenuType;
use Psr\Log\LoggerInterface;

/**
 * Migrates menu type configurations to group content menu types.
 */
final class MenuConfigMigrator implements MenuConfigMigratorInterface {

  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly LoggerInterface $logger,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public function migrateMenuTypeConfig(string $menuMachineName): void {
    try {
      $originalEntityType = $this->entityTypeManager->getStorage('menu')->load($menuMachineName);

      if (!$originalEntityType) {
        throw new \Exception("Menu type '$menuMachineName' not found.");
      }

      $groupContentMenuTypeStorage = $this->entityTypeManager->getStorage('group_content_menu_type');
      $groupContentMenuType = $groupContentMenuTypeStorage->load('group_' . $originalEntityType->id());

      if ($groupContentMenuType instanceof GroupContentMenuType) {
        return;
      }

      $newMenuBundle = $originalEntityType->createDuplicate();
      $newMenuBundle->set('entityTypeId', 'group_content_menu_type');
      $newMenuBundle->set('label', $originalEntityType->label());
      $newMenuBundle->set('id', 'group_' . $originalEntityType->id());
      $newMenuBundle->set('bundle_of', 'group_content_menu');
      $newMenuBundle->save();

      $this->copyFieldConfigurations(
        $originalEntityType->id(),
        'group_' . $originalEntityType->id()
      );
    }
    catch (\Throwable $exception) {
      $this->logger->error($exception->getMessage());
    }
  }

  /**
   * Copies field configurations from an original bundle to a new bundle.
   *
   * @param string $originalBundleId
   *   The ID of the original bundle.
   * @param string $newBundleId
   *   The ID of the new bundle.
   */
  private function copyFieldConfigurations(string $originalBundleId, string $newBundleId): void {
    $fieldConfigManager = $this->entityTypeManager->getStorage('field_config');

    $originalFieldConfigs = $fieldConfigManager->loadByProperties(['entity_type' => 'menu_link_content', 'bundle' => $originalBundleId]);

    foreach ($originalFieldConfigs as $fieldConfig) {
      $newFieldConfig = $fieldConfig->createDuplicate();
      $newFieldConfig->set('bundle', $newBundleId);
      $newFieldConfig->save();
    }
  }

}
