<?php

declare(strict_types=1);

namespace Drupal\sites_migrator\Drush\Commands;

use Drupal\Component\DependencyInjection\ContainerInterface;
use Drupal\sites_migrator\ContentInheritService;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;

/**
 * A Drush command class for content inheritance to sites. Especially for
 * existing Drupal sites, which receive a new sites group module setup.
 */
final class ContentInheritCommands extends DrushCommands {

  public function __construct(
    private readonly ContentInheritService $contentInheritService,
  ) {
    parent::__construct();
  }

  public static function create(ContainerInterface $container): self {
    return new static($container->get('sites_migrator.content_inherit_service'));
  }

  /**
   * Inherit content into the Drupal context. Handles Drupal content entity
   * types.
   */
  #[CLI\Command(name: 'sites_migrator:content-inherit', aliases: ['smci'])]
  #[CLI\Argument(name: 'groupId', description: 'The ID of the group.')]
  #[CLI\Usage(name: 'sites_migrator:content-inherit 1', description: 'Usage description')]
  public function inheritContent(?string $groupId = NULL): void {
    $this->contentInheritService->inheritContent($groupId, $this->output());
    $this->logger()->success('Inherited all content into sites.');
  }

  /**
   * Inherit content into the Drupal context. Handles Drupal content entity
   * types.
   */
  #[CLI\Command(name: 'sites_migrator:domain-content-inherit', aliases: ['smdci'])]
  #[CLI\Argument(name: 'domainId', description: 'The domain ID of the domain. E.g. "sites_domain_ddev_site".')]
  #[CLI\Argument(name: 'groupId', description: 'The ID of the group.')]
  #[CLI\Usage(name: 'sites_migrator:domain-content-inherit sites_domain_ddev_site 2', description: 'Usage description')]
  public function inheritDomainContent(string $domainId, string $groupId): void {
    $this->contentInheritService->inheritContent(
      groupId: $groupId,
      output: $this->output(),
      sourceContentConditions: ['field_domain_access' => $domainId]
    );
    $this->logger()->success('Inherited all content into sites.');
  }

}
