<?php

declare(strict_types=1);

namespace Drupal\sites_migrator\Drush\Commands;

use Drupal\Component\DependencyInjection\ContainerInterface;
use Drupal\sites_migrator\Menu\MenuConfigMigrator;
use Drupal\sites_migrator\Menu\MenuContentMigrator;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;

final class MenuMigrationCommands extends DrushCommands {

  public function __construct(
    private readonly MenuContentMigrator $menuContentMigrator,
    private readonly MenuConfigMigrator $menuConfigMigrator,
  ) {
    parent::__construct();
  }

  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('sites_migrator.menu_content_migrator'),
      $container->get('sites_migrator.menu_config_migrator')
    );
  }

  /**
   * Migrates a menu tree into group content menu.
   */
  #[CLI\Command(name: 'sites_migrator:migrate-menu-content', aliases: ['sm:migrate-menu-content'])]
  #[CLI\Argument(name: 'sourceMenuName', description: 'The machine name of the source Drupal core menu. E.g. "mie-demo-base-menu".')]
  #[CLI\Argument(name: 'groupId', description: 'The numeric ID of the group. E.g. "2".')]
  #[CLI\Argument(name: 'groupMenuBundle', description: 'The bundle of the group content menu. E.g. "basic_menu".')]
  #[CLI\Argument(name: 'menuLabel', description: 'The menu label. E.g. "Example Menu".')]
  #[CLI\Usage(name: 'sites_migrator:migrate-menu-content mie-demo-base-menu 2 basic_menu "Example Menu"', description: 'Usage description.')]
  public function migrateMenuContent(string $sourceMenuName, int $groupId, string $groupMenuBundle, string $menuLabel): void {
    try {
      $this->menuContentMigrator->migrateMenuItems($sourceMenuName, $groupId, $groupMenuBundle, $menuLabel, $this->output());
      $this->io()->success(dt('Menu content migrated.'));
    }
    catch (\Exception $e) {
      $this->logger()->error('Migration failed: ' . $e->getMessage());
      $this->io()->error('Migration failed: ' . $e->getMessage());
    }
  }

  #[CLI\Command(name: 'sites_migrator:migrate-domain-menu-content', aliases: ['sm:migrate-domain-menu-content'])]
  #[CLI\Argument(name: 'sourceMenuName', description: 'The machine name of the source Drupal core menu. E.g. "mie-demo-base-menu".')]
  #[CLI\Argument(name: 'groupId', description: 'The numeric ID of the group. E.g. "2".')]
  #[CLI\Argument(name: 'groupMenuBundle', description: 'The bundle of the group content menu. E.g. "basic_menu".')]
  #[CLI\Argument(name: 'menuLabel', description: 'The menu label. E.g. "Example Menu".')]
  #[CLI\Argument(name: 'domainId', description: 'The domain ID of the domain. E.g. "sites_domain_ddev_site".')]
  #[CLI\Usage(name: 'sites_migrator:migrate-domain-menu-content main 1 group_main "Example Menu" sites_domain_ddev_site', description: 'Usage description.')]
  public function migrateDomainMenuContent(
    string $sourceMenuName,
    int $groupId,
    string $groupMenuBundle,
    string $menuLabel,
    string $domainId,
  ): void {
    try {
      $this->menuContentMigrator->migrateDomainMenuItems(
        $sourceMenuName,
        $groupId,
        $groupMenuBundle,
        $menuLabel,
        $domainId,
        $this->output()
      );
      $this->io()->success(dt('Menu content migrated.'));
    }
    catch (\Exception $e) {
      $this->logger()->error('Migration failed: ' . $e->getMessage());
      $this->io()->error('Migration failed: ' . $e->getMessage());
    }
  }

  /**
   * Migrates a non-group menu config into a menu with leverages the
   * group_content_menu_bundles module. Fields from menu_item_extras module
   * will be also migrated.
   */
  #[CLI\Command(name: 'sites_migrator:migrate-menu-config', aliases: ['sm:migrate-menu-config'])]
  #[CLI\Argument(name: 'sourceMenuName', description: 'The machine name of the source Drupal core menu.')]
  #[CLI\Usage(name: 'sites_migrator:migrate-menu-config main', description: 'Usage description.')]
  public function migrateMenuConfig(string $sourceMenuName): void {
    try {
      $this->menuConfigMigrator->migrateMenuTypeConfig($sourceMenuName);
      $this->io()->success(dt('Menu config migrated. Do not forget to export your config.'));
    }
    catch (\Exception $e) {
      $this->logger()->error('Migration failed: ' . $e->getMessage());
      $this->io()->error('Migration failed: ' . $e->getMessage());
    }
  }

}
