<?php
declare(strict_types=1);

namespace Drupal\sites_migrator;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\group\Entity\GroupInterface;
use Drupal\group\Plugin\Group\Relation\GroupRelationInterface;
use Drupal\sites_group\SitesGroupServiceInterface;
use Symfony\Component\Console\Helper\ProgressBar;
use Symfony\Component\Console\Output\ConsoleOutputInterface;

/**
 * Inherits content entities from various bundles into the group context.
 */
final class ContentInheritService implements ContentInheritServiceInterface {

  public function __construct(
    readonly private SitesGroupServiceInterface $sitesGroupService,
    readonly public EntityTypeManagerInterface $entityTypeManager,
    readonly public LoggerChannelInterface $logger,
  ) {}

  public function inheritUsers(?ConsoleOutputInterface $output = NULL, ?array $sourceContentConditions = NULL): void {
    $group = $this->entityTypeManager->getStorage('group');
    $groups = $group->loadByProperties([
      'type' => 'site'
    ]);

    foreach ($groups as $group) {
      $group = $this->entityTypeManager->getStorage('group')->load($group->id());
      assert($group instanceof GroupInterface, 'Group not found');
      $userEntityIds = $this->fetchUserEntityIds($sourceContentConditions);
      $total = count($userEntityIds);
      $progressBar = NULL;
      if ($output) {
        $progressBar = new ProgressBar($output, $total);
        $progressBar->start();
      }
      foreach ($userEntityIds as $userId) {
        $user = $this->entityTypeManager->getStorage('user')->load($userId);

        // Do not inherit the anonymous user.
        if ($user->isAnonymous()) {
          continue;
        }

        try {
          $this->ensureSiteContextByGroup($user, $group->id(), FALSE);
        } catch (\Throwable $t) {
          $this->logger->error($t->getMessage());
        }
        if ($progressBar) {
          $progressBar->advance();
        }
      }
      if ($progressBar) {
        $progressBar->finish();
        $output->writeln('');
      }
    }
  }

  public function inheritContent(string|null $groupId, ?ConsoleOutputInterface $output = NULL, ?array $sourceContentConditions = NULL): void {
    $groupId = $this->handleGroupId($groupId);
    $group = $this->entityTypeManager->getStorage('group')->load($groupId);
    assert($group instanceof GroupInterface, 'Group not found');
    $contentEntityIds = $this->fetchContentEntityIds($group, $sourceContentConditions);
    $total = 0;
    foreach ($contentEntityIds as $entityIds) {
      $total += count($entityIds);
    }
    $progressBar = NULL;
    if ($output) {
      $progressBar = new ProgressBar($output, $total);
      $progressBar->start();
    }
    foreach ($contentEntityIds as $entityTypeId => $entityIds) {
      $contentEntityStorage = $this->entityTypeManager->getStorage($entityTypeId);
      foreach ($entityIds as $entity_id) {
        $contentEntity = $contentEntityStorage->load($entity_id);
        try {
          $this->ensureSiteContextByGroup($contentEntity, $groupId);
        } catch (\Throwable $t) {
          $this->logger->error($t->getMessage());
        }
        if ($progressBar) {
          $progressBar->advance();
        }
      }
    }
    if ($progressBar) {
      $progressBar->finish();
      $output->writeln('');
    }
  }

  public function ensureSiteContextByGroup(ContentEntityInterface $entity, string $groupId, bool $bypassUsers = TRUE): void {
    $group = $this->getGroup($groupId);
    if (!$this->groupIsSite($group)) {
      throw new \Exception('The group is not a site.');
    }
    $pluginId = $this->sitesGroupService->getPluginIdByEntity($entity);
    if ($pluginId === 'group_membership' && $bypassUsers === TRUE) {
      return;
    }
    if ($group instanceof GroupInterface && $pluginId !== NULL) {
      if ($this->isContentPluginInstalledForGroup($group, $pluginId)) {
        $relatedEntities = $group->getRelatedEntities($pluginId);
        $entityAlreadyRelated = FALSE;
        foreach ($relatedEntities as $relatedEntity) {
          if ($relatedEntity->id() === $entity->id() &&
            $relatedEntity->getEntityTypeId() === $entity->getEntityTypeId()) {
            $entityAlreadyRelated = TRUE;
            break;
          }
        }
        if (!$entityAlreadyRelated) {
          $group->addRelationship(
            $entity,
            $pluginId
          );
        }
      }
    }
  }

  private function handleGroupId(?string $groupId): string {
    if (is_string($groupId)) {
      return $groupId;
    }
    $groups = $this->getAllGroups();
    if (count($groups) === 0) {
      throw new \Exception('No groups available to inherit content.');
    }
    if (count($groups) > 1) {
      throw new \Exception('Multiple groups found. Please specify a group ID.');
    }
    return reset($groups)->id();
  }

  private function getGroup(string $groupId): ?GroupInterface {
    $group = $this->entityTypeManager->getStorage('group')->load($groupId);
    return $group instanceof GroupInterface ? $group : NULL;
  }

  /**
   * Fetches content entity IDs for all content types except users.
   */
  private function fetchContentEntityIds(GroupInterface $group, ?array $sourceContentConditions = NULL): array {
    $content_entity_ids = [];
    foreach ($group->getGroupType()->getInstalledPlugins() as $plugin) {
      assert($plugin instanceof GroupRelationInterface);
      $definition = $plugin->getPluginDefinition();
      if (!$entity_type_id = $definition->get('entity_type_id')) {
        continue;
      }
      // Skip user entities, as they are handled separately.
      if ($entity_type_id === 'user') {
        continue;
      }
      $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
      if ($entity_type->entityClassImplements(ContentEntityInterface::class)) {
        try {
          $storage = $this->entityTypeManager->getStorage($entity_type_id);
          $query = $storage->getQuery()->accessCheck(FALSE);
          if ($sourceContentConditions) {
            foreach ($sourceContentConditions as $name => $value) {
              $query->condition($name, $value);
            }
          }
          $entity_ids = $query->execute();
          if (!empty($entity_ids)) {
            $content_entity_ids[$entity_type_id] = $entity_ids;
          }
        } catch (\Exception $e) {
          $this->logger->error('Error fetching @type entities: @message', [
            '@type' => $entity_type_id,
            '@message' => $e->getMessage(),
          ]);
        }
      }
    }
    return $content_entity_ids;
  }

  /**
   * Fetches user entity IDs.
   */
  private function fetchUserEntityIds(?array $sourceContentConditions = NULL): array {
    try {
      $storage = $this->entityTypeManager->getStorage('user');
      $query = $storage->getQuery()->accessCheck(FALSE);
      if ($sourceContentConditions) {
        foreach ($sourceContentConditions as $name => $value) {
          $query->condition($name, $value);
        }
      }
      return $query->execute();
    } catch (\Exception $e) {
      $this->logger->error('Error fetching user entities: @message', [
        '@message' => $e->getMessage(),
      ]);
      return [];
    }
  }

  private function getAllGroups(): array {
    $storage = $this->entityTypeManager->getStorage('group');
    $group_ids = $storage->getQuery()->accessCheck(FALSE)->execute();
    $groupsAsSites = [];
    if (!empty($group_ids)) {
      $allGroups = $storage->loadMultiple($group_ids);
      foreach ($allGroups as $group) {
        if ($this->groupIsSite($group)) {
          $groupsAsSites[] = $group;
        }
      }
    }
    return $groupsAsSites;
  }
  
  private function groupIsSite(GroupInterface $group): bool {
    return array_key_exists($group->bundle(), $this->getGroupTypes());
  }

  private function getGroupTypes(): array {
    $group_types = [];

    foreach ($this->entityTypeManager->getStorage('group_type')->loadMultiple() as $group_type) {
      if ($group_type->getThirdPartySetting('sites_group', 'status')) {
        $group_types[$group_type->id()] = $group_type;
      }
    }

    return $group_types;
  }

  private function isContentPluginInstalledForGroup(GroupInterface $group, string $pluginId): bool {
    $groupType = $group->getGroupType();
    $installedPlugins = $groupType->getInstalledPlugins();

    return $installedPlugins->has($pluginId);
  }

}
