<?php

declare(strict_types=1);

namespace Drupal\sites_migrator;

use Drupal\domain\Entity\Domain;
use Drupal\node\Entity\Node;
use Symfony\Component\Console\Helper\ProgressBar;
use Symfony\Component\Console\Output\ConsoleOutput;

final readonly class DomainModuleService {

  /**
   * @param \Drupal\sites_migrator\ContentInheritService $parent
   *   The parent content entity service.
   */
  public function __construct(
    private ContentInheritService $parent,
  ) {}

  public function inheritDomainContent(?ConsoleOutput $output = NULL): void {
    $nodeQuery = $this->parent->entityTypeManager
      ->getStorage('node')
      ->getQuery()
      ->condition('field_domain_access', NULL, 'IS NOT NULL')
      ->accessCheck(FALSE);
    $nids = $nodeQuery->execute();

    $progressBar = NULL;
    if ($output) {
      $progressBar = new ProgressBar($output, count($nids));
      $progressBar->start();
    }

    foreach ($nids as $nid) {
      $node = Node::load($nid);

      $domainId = $node->get('field_domain_access')->getString();

      $domainQuery = $this->parent->entityTypeManager
        ->getStorage('domain')
        ->getQuery()
        ->condition('id', $domainId)
        ->accessCheck(FALSE);
      $domainIds = $domainQuery->execute();

      if (count($domainIds) === 0) {
        if ($progressBar) {
          $progressBar->advance();
        }
        continue;
      }

      $domain = Domain::load(current($domainIds));
      if (!$domain) {
        if ($progressBar) {
          $progressBar->advance();
        }
        continue;
      }

      $groupQuery = $this->parent->entityTypeManager
        ->getStorage('group')
        ->getQuery()
        ->condition('field_sites_hostnames', $domain->getHostname())
        ->accessCheck(FALSE);
      $gidResult = $groupQuery->execute();

      if (!$gidResult) {
        $output->writeln('Could not find group with "field_sites_hostnames" value ' . $domain->getHostname() . '. You may create the related group?');
        $progressBar->advance();
        continue;
      }

      if (count($gidResult) === 1) {
        $gid = current($gidResult);
        $this->parent->ensureSiteContextByGroup($node, $gid);
      }

      if ($progressBar) {
        $progressBar->advance();
      }
    }

    if ($progressBar) {
      $progressBar->finish();
      $output->writeln('');
    }
  }

}
