<?php

namespace Drupal\skynetaccessibility_scanner\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for Skynet Accessibility Scanner dashboard.
 */
class SkynetAccessibilityScannerController extends ControllerBase {

  /**
   * The module extension list service.
   *
   * @var \Drupal\skynetaccessibility_scanner\Core\Extension\ModuleExtensionList
   */
  protected $moduleExtensionList;

  /**
   * The file URL generator service.
   *
   * @var \Drupal\skynetaccessibility_scanner\Core\File\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Logger channel factory service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a MonitoringController object.
   */
  public function __construct(ModuleExtensionList $moduleExtensionList, FileUrlGeneratorInterface $fileUrlGenerator, RequestStack $requestStack, LoggerChannelFactoryInterface $loggerFactory) {
    $this->moduleExtensionList = $moduleExtensionList;
    $this->fileUrlGenerator = $fileUrlGenerator;
    $this->requestStack = $requestStack;
    $this->loggerFactory = $loggerFactory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('extension.list.module'),
      $container->get('file_url_generator'),
      $container->get('request_stack'),
      $container->get('logger.factory')
    );
  }

  /**
   * Make a cURL POST request helper.
   */
  private function curlPost($url, $params = [], $isJson = FALSE) {
    $ch = curl_init($url);

    $headers = ['Cache-Control: no-cache', 'Pragma: no-cache'];
    if ($isJson) {
      $payload = json_encode($params);
      $headers[] = 'Content-Type: application/json';
      curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
      curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
    }
    else {
      curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
      curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($params));
    }

    curl_setopt($ch, CURLOPT_RETURNTRANSFER, TRUE);
    curl_setopt($ch, CURLOPT_POST, TRUE);
    // Add timeout.
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);

    $response = curl_exec($ch);
    if (curl_errno($ch)) {
      $this->loggerFactory->get('skynetaccessibility_scanner')->error('cURL error: @err', ['@err' => curl_error($ch)]);
      curl_close($ch);
      return [];
    }

    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($http_code != 200) {
      $this->loggerFactory->get('skynetaccessibility_scanner')->error('API returned HTTP code: @code', ['@code' => $http_code]);
      return [];
    }

    return json_decode($response, TRUE);
  }

  /**
   * Build the scanning dashboard render array.
   */
  public function getScanDetails() {

    // Get module path and URL.
    $module_path = $this->moduleExtensionList->getPath('skynetaccessibility_scanner');

    // Normalize case and avoid duplicate directories.
    $clean_path = str_ireplace(['/Images', '/images'], '', $module_path);
    $module_url = $this->fileUrlGenerator->generateAbsoluteString($clean_path . '/images');

    $data = [];
    $plans = [];
    $scan_details = [];
    $violationLink = '#';

    try {
      // Resolve domain dynamically (with optional ?site= override for QA)
     // $domain_name = 'drupal.skynettechnologies.us';
       $request = $this->requestStack->getCurrentRequest();
       $domain_name = $request->query->get('site');
      if (empty($domain_name)) {
        $domain_name = $request->server->get('SERVER_NAME') ?? $request->getHost();
      }
      $domain_name = strtolower(preg_replace('/^www\\./i', '', $domain_name ?? ''));

      $this->loggerFactory->get('skynetaccessibility_scanner')->debug('Fetching fresh data for domain: @domain', ['@domain' => $domain_name]);

      // API 1: Register domain.
      $register_data = $this->curlPost(
        'https://skynetaccessibilityscan.com/api/register-domain-platform',
        [
          'website' => base64_encode($domain_name),
          'platform' => 'drupal',
          'is_trial_period' => 1,
          'name' => $domain_name,
          'email' => 'no-reply@' . $domain_name,
          'comapany_name' => $domain_name,
          'package_type' => '25-pages',
        ]
      );

      // Basic check.
      if (empty($register_data) || (isset($register_data['success']) && $register_data['success'] === FALSE)) {
        $this->loggerFactory
          ->get('skynetaccessibility_scanner')
          ->warning(
            'Domain registration may have failed for @domain. Response: @resp',
            [
              '@domain' => $domain_name,
              '@resp' => print_r($register_data, TRUE),
            ]
          );
      }
      else {
        // If API returned a website id, expose it for downstream calls.
        if (!empty($register_data['data']['website_id'])) {
          $data['website_id'] = $register_data['data']['website_id'];
        }
      }

      // API 2: scan details.
      $scan_details = $this->curlPost(
        'https://skynetaccessibilityscan.com/api/get-scan-detail',
        ['website' => base64_encode($domain_name)]
      );

      $this->loggerFactory->get('skynetaccessibility_scanner')->debug('Scan details response: @response', ['@response' => print_r($scan_details, TRUE)]);

      if (!empty($scan_details['data'][0])) {
        $row = $scan_details['data'][0];
        $data['domain'] = $row['domain'] ?? $domain_name;
        $data['fav_icon'] = $row['fav_icon'] ?? '';
        $data['url_scan_status'] = $row['url_scan_status'] ?? 0;
        $data['scan_status'] = $row['scan_status'] ?? 0;
        $data['total_selected_pages'] = $row['total_selected_pages'] ?? 0;
        $data['total_last_scan_pages'] = $row['total_last_scan_pages'] ?? 0;
        $data['total_pages'] = $row['total_pages'] ?? 0;
        $data['last_url_scan'] = $row['last_url_scan'] ?? 0;
        $data['total_scan_pages'] = $row['total_scan_pages'] ?? 0;
        $data['last_scan'] = $row['last_scan'] ?? NULL;
        $data['next_scan_date'] = $row['next_scan_date'] ?? NULL;
        $data['success_percentage'] = $row['success_percentage'] ?? '0';
        $data['scan_violation_total'] = $row['scan_violation_total'] ?? '0';
        $data['total_violations'] = $row['total_violations'] ?? 0;
        $data['package_name'] = $row['name'] ?? '';
        $data['package_id'] = $row['package_id'] ?? '';
        $data['page_views'] = $row['page_views'] ?? '';
        $data['package_price'] = $row['package_price'] ?? '';
        $data['subscr_interval'] = $row['subscr_interval'] ?? '';
        $data['end_date'] = $row['end_date'] ?? '';
        $data['website_id'] = $row['website_id'] ?? '';
        $data['dashboard_link'] = $scan_details['dashboard_link'] ?? '';
        $data['total_fail_sum'] = $row['total_fail_sum'] ?? '';

        // Add missing variables from API response.
        $data['final_price'] = $row['final_price'] ?? 0;
        $data['price'] = $row['price'] ?? 0;
        $data['payment_status'] = $row['payment_status'] ?? '';
        $data['start_date'] = $row['start_date'] ?? '';
        $data['cancel_date'] = $row['cancel_date'] ?? '';
        $data['cancel_reason_id'] = $row['cancel_reason_id'] ?? 0;
        $data['cancel_reason_description'] = $row['cancel_reason_description'] ?? '';
        $data['is_trial_period'] = $row['is_trial_period'] ?? 0;
        $data['is_pre_trial'] = $row['is_pre_trial'] ?? 0;
        $data['is_cancel_offer_accept'] = $row['is_cancel_offer_accept'] ?? 0;
        $data['coupon_id'] = $row['coupon_id'] ?? 0;
        $data['api_key'] = $row['api_key'] ?? '';
        $data['scanner_plan'] = $row['scanner_plan'] ?? 0;
        $data['is_main'] = $row['is_main'] ?? 0;
        $data['frequency'] = $row['frequency'] ?? 1;
        $data['preferred_day_for_scanning'] = $row['preferred_day_for_scanning'] ?? '';
        $data['send_email_notifications'] = $row['send_email_notifications'] ?? 0;
        $data['new_error_pdf_attachment_on_email'] = $row['new_error_pdf_attachment_on_email'] ?? 0;
        $data['is_pause_scanning'] = $row['is_pause_scanning'] ?? 0;
        $data['scan_start'] = $row['scan_start'] ?? 0;
        $data['parent_id'] = $row['parent_id'] ?? 0;
        $data['scan_success_total'] = $row['scan_success_total'] ?? 0;
        $data['paypal_subscr_id'] = $row['paypal_subscr_id'] ?? '';
        $data['package_monthly_price'] = $row['package_monthly_price'] ?? 0;
        $data['product_type'] = $row['product_type'] ?? 0;
        $data['slug'] = $row['slug'] ?? '';
        $data['id'] = $row['id'] ?? 0;
        $data['url'] = $row['url'] ?? '';
        $data['widget_purchased'] = $row['widget_purchased'] ?? 0;
        $data['created_at'] = $row['created_at'] ?? '';
        $data['is_url_scan_now'] = $row['is_url_scan_now'] ?? 0;
        $data['used_canceled_discount'] = $row['used_canceled_discount'] ?? 0;
        $data['remediation'] = $row['remediation'] ?? 0;
        $data['page_id'] = $row['page_id'] ?? 0;
        $data['total_success_sum'] = $row['total_success_sum'] ?? 0;
        $data['fail_percentage'] = $row['fail_percentage'] ?? 0;
        $data['is_expired'] = $row['is_expired'] ?? 0;
        $data['total_scan_monthly_pages'] = $row['total_scan_monthly_pages'] ?? 0;
        $data['total_scanning_process'] = $row['total_scanning_process'] ?? 0;
      }

      // API 3: scan count.
      $scan_count = $this->curlPost(
        'https://skynetaccessibilityscan.com/api/get-scan-count',
        ['website' => base64_encode($domain_name)]
      );

      $widget_flg = $scan_count['widget_purchased'] ?? 0;
      $details = $scan_count['scan_details'] ?? [];

      // Detect if scan_details has nested keys.
      if (isset($details['with_remediation']) || isset($details['without_remediation'])) {
        $with_rem = $details['with_remediation'] ?? [];
        $without_rem = $details['without_remediation'] ?? [];
      }
      else {
        // Handle flat structure.
        $with_rem = $details;
        $without_rem = $details;
      }

      $data['scan_details_with_remediation'] = $with_rem;
      $data['scan_details_without_remediation'] = $without_rem;

      // Pick correct one safely.
      $data['scan_details'] = (!empty($widget_flg) && (int) $widget_flg === 1)
        ? $with_rem
        : $without_rem;

      // API 4: packages list.
      $package_list = $this->curlPost(
        'https://skynetaccessibilityscan.com/api/packages-list',
        ['website' => base64_encode($domain_name)],
        TRUE
      );

      $websiteId = (string) ($data['website_id'] ?? '');
      $activePackageId = $data['package_id'] ?? '';
      $activeInterval = $data['subscr_interval'] ?? 'M';
      $allowedNames = ['Small Site', 'Medium Site', 'Large Site', 'Extra Large Site'];

      // Initialize plan status variables.
      $is_trial = $data['is_trial_period'] ?? FALSE;
      $is_canceled = !empty($data['cancel_date']);
      // Default.
      $plan_status = 'active';

      $today = date('Y-m-d');
      $end_date_str = !empty($data['end_date']) ? date('Y-m-d', strtotime($data['end_date'])) : NULL;
      $is_expired = $data['is_expired'] ?? FALSE;
      $renewal_text = 'Renews on:';
      $button_text = 'Cancel Subscription';
      $plan_status = 'active';

      if ($is_expired || ($end_date_str && $end_date_str < $today)) {
        $is_expired = TRUE;
        $plan_status = 'expired';
        $renewal_text = 'Expired on:';
        $button_text = 'Renew Plan';
      }

      // $current_plan_data removed (unused)
      if (!empty($package_list['current_active_package'])) {
        // matchedWebsite removed (unused)
        foreach ($package_list['current_active_package'] as $key => $package) {
          if ((string) $key === $websiteId) {
            // matchedWebsite flag removed.
            // Merge API and fallback values (handle variants and synonyms)
            $payment_status_raw = $package['payment_status'] ?? $data['payment_status'] ?? '';
            $payment_status = strtolower(trim($payment_status_raw));
            $is_trial_period =
              !empty($package['is_trial_period'])
              || !empty($data['is_trial_period'])
              || in_array(
                strtolower($package['status'] ?? ''),
                ['trial', 'trialing'],
                TRUE
              );
            $end_date = $package['end_date'] ?? $package['current_period_end'] ?? $data['end_date'] ?? '';
            $canceled_at = $package['canceled_at'] ?? $package['cancelled_at'] ?? $data['cancel_date'] ?? $data['cancelled_date'] ?? '';
            $is_expired_api = (int) ($package['is_expired'] ?? $data['is_expired'] ?? 0);

            // Default.
            $plan_status = 'active';
            $renewal_text = 'Renews on:';
            $button_text = 'Cancel Subscription';

            // Normalize date comparison.
            $today = strtotime('today');
            $end_date_ts = !empty($end_date) ? strtotime($end_date) : 0;

            // --- Priority order of plan state detection ---
            if (
              !empty($is_expired_api)
              || ($end_date_ts && $end_date_ts < $today)
              || in_array($payment_status, ['expired', 'past_due'], TRUE)
            ) {
              $plan_status = 'expired';
              $renewal_text = 'Expired on:';
              $button_text = 'Renew Plan';
            }
            elseif (
              in_array(
                $payment_status,
                [
                  'canceled',
                  'cancelled',
                  'cancel',
                  'cancellation',
                  'cancelled_by_user',
                  'inactive',
                  'terminated',
                ],
                TRUE
              )
              || !empty($canceled_at)
            ) {

              $plan_status = 'canceled';
              $renewal_text = 'Canceled on:';
              $button_text = 'Renew Plan';
            }
            elseif ($is_trial_period) {
              $plan_status = 'trial';
              $renewal_text = 'Expires on:';
              $button_text = 'Cancel Trial';
            }

            // Save for Twig.
            $data['plan_status'] = $plan_status;
            $data['renewal_text'] = $renewal_text;
            $data['button_text'] = $button_text;

            // Expose important dates for display.
            $data['plan_end_date'] = $end_date;
            $data['plan_canceled_at'] = $canceled_at;

            // Compute the correct plan action link.
            $plan_action_link = $data['dashboard_link'] ?? '#';
            try {
              if ($plan_status === 'expired' || $plan_status === 'canceled') {
                $renew_link_data = $this->curlPost(
                  'https://skynetaccessibilityscan.com/api/generate-plan-action-link',
                  [
                    'package_id' => $activePackageId,
                    'current_package_id' => $activePackageId,
                    'interval' => $activeInterval,
                    'website_id' => $websiteId,
                    'action' => 'renew',
                  ]
                );
                $plan_action_link = $renew_link_data['action_link'] ?? $plan_action_link;
              }
              else {
                $cancel_link_data = $this->curlPost(
                  'https://skynetaccessibilityscan.com/api/generate-plan-action-link',
                  [
                    'package_id' => $activePackageId,
                    'current_package_id' => $activePackageId,
                    'interval' => $activeInterval,
                    'website_id' => $websiteId,
                    'action' => 'cancel',
                  ]
                              );
                $plan_action_link = $cancel_link_data['action_link'] ?? $plan_action_link;
              }
            }
            catch (\Exception $ex) {
              $this->loggerFactory->get('skynetaccessibility_scanner')->warning('Could not generate plan action link: @msg', ['@msg' => $ex->getMessage()]);
            }

            $data['plan_action_link'] = $plan_action_link;

            // Debug log to verify API behavior.
            $this->loggerFactory->get('skynetaccessibility_scanner')->debug(
              'Plan status computed => Status: @status | Payment: @pay | Expired: @exp | Canceled: @can | Trial: @trial | End: @end',
              [
                '@status' => $plan_status,
                '@pay' => $payment_status,
                '@exp' => $is_expired_api,
                '@can' => $canceled_at,
                '@trial' => $is_trial_period,
                '@end' => $end_date,
              ]
            );

            break;
          }
        }
      }

      // If no websiteId match found, compute plan state from the first package.
      if (!isset($data['plan_status']) && !empty($package_list['current_active_package'])) {
        $firstKeyTmp = array_key_first($package_list['current_active_package']);
        $fallbackPkg = $package_list['current_active_package'][$firstKeyTmp] ?? [];
        $payment_status_fb = strtolower(trim($fallbackPkg['payment_status'] ?? ($data['payment_status'] ?? '')));
        $is_trial_period_fb =
          !empty($fallbackPkg['is_trial_period'])
          || !empty($data['is_trial_period'])
          || in_array(
            strtolower($fallbackPkg['status'] ?? ''),
            ['trial', 'trialing'],
            TRUE
          );

        $end_date_fb = $fallbackPkg['end_date'] ?? $fallbackPkg['current_period_end'] ?? ($data['end_date'] ?? '');
        $canceled_at_fb = $fallbackPkg['canceled_at'] ?? $fallbackPkg['cancelled_at'] ?? ($data['cancel_date'] ?? ($data['cancelled_date'] ?? ''));
        $is_expired_fb = (int) ($fallbackPkg['is_expired'] ?? ($data['is_expired'] ?? 0));

        $computed_status = 'active';
        $computed_renewal_text = 'Renews on:';
        $computed_button_text = 'Cancel Subscription';

        $todayTs = strtotime('today');
        $endTsFb = !empty($end_date_fb) ? strtotime($end_date_fb) : 0;

        if (
          !empty($is_expired_fb)
          || ($endTsFb && $endTsFb < $todayTs)
          || in_array(
            $payment_status_fb,
            ['expired', 'past_due'],
            TRUE
          )
        ) {

          $computed_status = 'expired';
          $computed_renewal_text = 'Expired on:';
          $computed_button_text = 'Renew Plan';
        }
        elseif (
          in_array(
            $payment_status_fb,
            [
              'canceled',
              'cancelled',
              'cancel',
              'cancellation',
              'cancelled_by_user',
              'inactive',
              'terminated',
            ],
            TRUE
          )
          || !empty($canceled_at_fb)
        ) {
          $computed_status = 'canceled';
          $computed_renewal_text = 'Canceled on:';
          $computed_button_text = 'Renew Plan';
        }
        elseif ($is_trial_period_fb) {
          $computed_status = 'trial';
          $computed_renewal_text = 'Expires on:';
          $computed_button_text = 'Cancel Trial';
        }

        $data['plan_status'] = $computed_status;
        $data['renewal_text'] = $computed_renewal_text;
        $data['button_text'] = $computed_button_text;
        $data['plan_end_date'] = $end_date_fb;
        $data['plan_canceled_at'] = $canceled_at_fb;

        // Action link for fallback.
        $plan_action_link_fb = $data['dashboard_link'] ?? '#';
        try {
          if ($computed_status === 'expired' || $computed_status === 'canceled') {
            $renew_link_fb = $this->curlPost('https://skynetaccessibilityscan.com/api/generate-plan-action-link', [
              'package_id' => $activePackageId,
              'current_package_id' => $activePackageId,
              'interval' => $activeInterval,
              'website_id' => $websiteId,
              'action' => 'renew',
            ]);
            $plan_action_link_fb = $renew_link_fb['action_link'] ?? $plan_action_link_fb;
          }
          else {
            $cancel_link_fb = $this->curlPost('https://skynetaccessibilityscan.com/api/generate-plan-action-link', [
              'package_id' => $activePackageId,
              'current_package_id' => $activePackageId,
              'interval' => $activeInterval,
              'website_id' => $websiteId,
              'action' => 'cancel',
            ]);
            $plan_action_link_fb = $cancel_link_fb['action_link'] ?? $plan_action_link_fb;
          }
        }
        catch (\Exception $ex) {
          $this->loggerFactory->get('skynetaccessibility_scanner')->warning('Fallback action link error: @msg', ['@msg' => $ex->getMessage()]);
        }
        $data['plan_action_link'] = $data['plan_action_link'] ?? $plan_action_link_fb;

        $this->loggerFactory->get('skynetaccessibility_scanner')->debug('Fallback plan status used => Status: @status | Payment: @pay | Expired: @exp | Canceled: @can | Trial: @trial | End: @end', [
          '@status' => $data['plan_status'],
          '@pay' => $payment_status_fb,
          '@exp' => $is_expired_fb,
          '@can' => $canceled_at_fb,
          '@trial' => $is_trial_period_fb,
          '@end' => $end_date_fb,
        ]);
      }

      $data1 = $package_list['current_active_package'] ?? [];
      $firstKey = $data1 ? array_key_first($data1) : NULL;
      $finalPrice = $firstKey ? ($data1[$firstKey]['final_price'] ?? $data['final_price'] ?? 0) : ($data['final_price'] ?? 0);
      $data['final_price'] = $finalPrice;

      // Icon map - update to use your actual image filenames.
      $iconMap = [
        'Small Site' => 'diamond.svg',
        'Medium Site' => 'pentagon.svg',
        'Large Site' => 'hexagon.svg',
        'Extra Large Site' => 'octagon.svg',
      ];

      // Background images map.
      $backgroundImages = [
        'sitemap' => 'sitemap-bg.png',
        'plan' => 'plan-bg.png',
        'pricing' => 'pricing-bg.png',
      ];

      // Process plans.
      if (!empty($package_list['Data'])) {
        foreach ($package_list['Data'] as $plan) {
          $planName = $plan['name'] ?? '';
          if (!$planName || !in_array($planName, $allowedNames)) {
            continue;
          }

          $packageId = $plan['id'] ?? NULL;
          if (!$packageId) {
            continue;
          }

          // Default values.
          $action = 'upgrade';
          $button_label = 'Upgrade';
          $is_plan_active = ($packageId == $activePackageId);
          $plan_status_display = 'available';

          // ✅ FIXED: Determine correct action for current plan
          if ($is_plan_active) {
            if ($plan_status === 'expired' || $plan_status === 'canceled') {
              // For expired/canceled plans, use 'upgrade' to get renewal link
              $action = 'upgrade';
              $button_label = 'Renew Plan';
              $plan_status_display = 'expired';
            } elseif ($plan_status === 'trial') {
              $action = 'cancel';
              $button_label = 'Cancel Trial';
              $plan_status_display = 'trial';
            } else {
              $action = 'cancel';
              $button_label = 'Cancel';
              $plan_status_display = 'active';
            }
          }

          // ✅ FIXED: Generate correct dashboard link
          $dashboardLink = '#';
          try {
            $link_data = $this->curlPost(
              'https://skynetaccessibilityscan.com/api/generate-plan-action-link',
              [
                'package_id' => $packageId,
                'current_package_id' => $activePackageId,
                'interval' => $activeInterval,
                'website_id' => $websiteId,
                'action' => $action,
              ]
            );

            if (!empty($link_data['action_link']) && $link_data['action_link'] !== '#') {
              $dashboardLink = $link_data['action_link'];
            } else {
              $dashboardLink = $data['dashboard_link'] ?? '#';
            }
          } catch (\Exception $ex) {
            $this->loggerFactory->get('skynetaccessibility_scanner')->warning('Plan action link generation failed for package @pkg: @msg', [
              '@pkg' => $packageId,
              '@msg' => $ex->getMessage()
            ]);
            $dashboardLink = $data['dashboard_link'] ?? '#';
          }

          // Only fetch violation link once
          if ($violationLink === '#') {
            try {
              $violation_data = $this->curlPost(
                'https://skynetaccessibilityscan.com/api/generate-plan-action-link',
                [
                  'website_id' => $websiteId,
                  'current_package_id' => $activePackageId,
                  'action' => 'violation',
                ]
              );
              $violationLink = $violation_data['action_link'] ?? $violationLink;
            } catch (\Exception $ex) {
              $this->loggerFactory->get('skynetaccessibility_scanner')->warning('Violation link generation failed: @msg', ['@msg' => $ex->getMessage()]);
            }
          }

          // Icon map
          $iconFilename = $iconMap[$planName] ?? 'default.svg';
          $iconUrl = $module_url . '/' . $iconFilename;

          $is_plan_trial = stripos($planName, 'small') !== FALSE ||
            ($plan['monthly_price'] ?? 0) <= 9.99;

          $paypalSubId = $data['paypal_subscr_id'] ?? '';

          // Final plan array
          $plans[] = [
            'id' => $packageId,
            'name' => $planName,
            'pages' => (int) ($plan['page_views'] ?? 0),
            'price_monthly' => $plan['monthly_price'] ?? 0,
            'strick_monthly_price' => $plan['strick_monthly_price'] ?? 0,
            'price_yearly' => $plan['price'] ?? 0,
            'strick_price' => $plan['strick_price'] ?? 0,
            'is_active' => $is_plan_active,
            'dashboard_link' => $dashboardLink,
            'action' => $action,
            'button_label' => $button_label,
            'final_price' => $finalPrice,
            'icon' => $iconUrl,
            'is_trial' => $is_plan_trial,
            'status' => $plan_status_display,
            'paypal_sub_id' => $paypalSubId,
            'is_current' => $is_plan_active,
          ];
        }
      }

      $data['plans'] = $plans;
      $data['violation_link'] = $violationLink;
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('skynetaccessibility_scanner')->error('Error: @msg', ['@msg' => $e->getMessage()]);
      $data['error'] = $e->getMessage();
    }

    // Compute plan status purely from scan-detail fields if still not set.
    if (!isset($data['plan_status']) || empty($data['plan_status'])) {
      $payment_status_fb2 = strtolower(trim($data['payment_status'] ?? ''));
      $is_trial_fb2 = !empty($data['is_trial_period']) || in_array($payment_status_fb2, ['trial', 'trialing'], TRUE);
      $end_fb2 = $data['end_date'] ?? '';
      $canceled_fb2 = $data['cancel_date'] ?? '';
      $is_expired_fb2 = (int) ($data['is_expired'] ?? 0);

      $computed_status2 = 'active';
      $computed_renewal_text2 = 'Renews on:';
      $computed_button_text2 = 'Cancel Subscription';

      $todayTs2 = strtotime('today');
      $endTs2 = !empty($end_fb2) ? strtotime($end_fb2) : 0;

      if (
        !empty($is_expired_fb2)
        || ($endTs2 && $endTs2 < $todayTs2)
        || in_array(
          $payment_status_fb2,
          ['expired', 'past_due'],
          TRUE
        )
      ) {

        $computed_status2 = 'expired';
        $computed_renewal_text2 = 'Expired on:';
        $computed_button_text2 = 'Renew Plan';
      }
      elseif (
        in_array(
          $payment_status_fb2,
          [
            'canceled',
            'cancelled',
            'cancel',
            'cancellation',
            'cancelled_by_user',
            'inactive',
            'terminated',
          ],
          TRUE
        )
        || !empty($canceled_fb2)
      ) {
        $computed_status2 = 'canceled';
        $computed_renewal_text2 = 'Canceled on:';
        $computed_button_text2 = 'Renew Plan';
      }
      elseif ($is_trial_fb2) {
        $computed_status2 = 'trial';
        $computed_renewal_text2 = 'Expires on:';
        $computed_button_text2 = 'Cancel Trial';
      }

      $data['plan_status'] = $computed_status2;
      $data['renewal_text'] = $computed_renewal_text2;
      $data['button_text'] = $computed_button_text2;
      $data['plan_end_date'] = $end_fb2;
      $data['plan_canceled_at'] = $canceled_fb2;

      // Generate action link accordingly.
      $plan_action_link_fb2 = $data['dashboard_link'] ?? '#';
      try {
        if ($computed_status2 === 'expired' || $computed_status2 === 'canceled') {
          $renew_link_fb2 = $this->curlPost('https://skynetaccessibilityscan.com/api/generate-plan-action-link', [
            'package_id' => $data['package_id'] ?? '',
            'current_package_id' => $data['package_id'] ?? '',
            'interval' => $data['subscr_interval'] ?? 'M',
            'website_id' => (string) ($data['website_id'] ?? ''),
            'action' => 'upgrade',
          ]);
          $plan_action_link_fb2 = $renew_link_fb2['action_link'] ?? $plan_action_link_fb2;
        }
        else {
          $cancel_link_fb2 = $this->curlPost('https://skynetaccessibilityscan.com/api/generate-plan-action-link', [
            'package_id' => $data['package_id'] ?? '',
            'current_package_id' => $data['package_id'] ?? '',
            'interval' => $data['subscr_interval'] ?? 'M',
            'website_id' => (string) ($data['website_id'] ?? ''),
            'action' => 'cancel',
          ]);
          $plan_action_link_fb2 = $cancel_link_fb2['action_link'] ?? $plan_action_link_fb2;
        }
      }
      catch (\Exception $ex) {
        $this->loggerFactory->get('skynetaccessibility_scanner')->warning('Final fallback action link error: @msg', ['@msg' => $ex->getMessage()]);
      }
      $data['plan_action_link'] = $data['plan_action_link'] ?? $plan_action_link_fb2;

      $this->loggerFactory->get('skynetaccessibility_scanner')->debug('Final fallback plan status => Status: @status | Payment: @pay | Expired: @exp | Canceled: @can | Trial: @trial | End: @end', [
        '@status' => $data['plan_status'],
        '@pay' => $payment_status_fb2,
        '@exp' => $is_expired_fb2,
        '@can' => $canceled_fb2,
        '@trial' => $is_trial_fb2,
        '@end' => $end_fb2,
      ]);
    }

    // Ensure local plan_status reflects final computed value.
    $plan_status = $data['plan_status'] ?? $plan_status;

    // Determine scanning status based on plan status and API pause flag.
    $scanning_paused = FALSE;
    $scan_pause_reason = '';

    if ($plan_status === 'expired') {
      $scanning_paused = TRUE;
      $scan_pause_reason = 'Plan Expired';
    }
    elseif ($plan_status === 'canceled') {
      $scanning_paused = TRUE;
      $scan_pause_reason = 'Plan Canceled';
    }
    elseif (($data['is_pause_scanning'] ?? 0) == 1) {
      $scanning_paused = TRUE;
      $scan_pause_reason = 'Scanning Paused';
    }

    $scan_status = $data['scan_status'] ?? 0;
    $success_percentage = (int) ($data['success_percentage'] ?? 0);

    // Override scan status text and icon if scanning is paused.
    if ($scanning_paused) {
      $scan_status_text = 'Paused';
      $scan_status_icon = '⏸️';
    }
    else {
      switch ($scan_status) {
        case 1:
          $scan_status_text = 'In Progress';
          $scan_status_icon = '⏳';
          break;

        case 2:
          $scan_status_text = 'Completed';
          $scan_status_icon = '✅';
          break;

        default:
          $scan_status_text = 'Not Started';
          $scan_status_icon = '⚠️';
          break;
      }
    }

    // Determine accessibility score display.
    $display_score = $success_percentage > 0 ? $success_percentage : 0;
    $display_score_value = $scanning_paused ? 'N/A' : $display_score;

    if ($scanning_paused) {
      $statusClass = 'scan-paused';
      $statusText = 'Scanning Paused';
    }
    else {
      if ($display_score <= 49) {
        $statusClass = 'not-compliant';
        $statusText = 'Not Compliant';
      }
      elseif ($display_score <= 84) {
        $statusClass = 'semi-compliant';
        $statusText = 'Semi Compliant';
      }
      else {
        $statusClass = 'compliant';
        $statusText = 'Compliant';
      }
    }

    // Compute plan date to display based on plan status.
    $ps = $data['plan_status'] ?? $plan_status;
    $plan_end_date_raw = $data['plan_end_date'] ?? ($data['end_date'] ?? '');
    $plan_canceled_at_raw = $data['plan_canceled_at'] ?? ($data['cancel_date'] ?? '');
    $plan_date_raw = ($ps === 'canceled') ? $plan_canceled_at_raw : $plan_end_date_raw;
    $plan_display_date = !empty($plan_date_raw) ? date("F j, Y", strtotime($plan_date_raw)) : 'N/A';

    // Create the render array with proper cache disabling.
    $build = [
      '#theme' => 'scanning_dashboard',
      '#module_url' => $module_url,
      '#module_path' => $module_path,
      '#background_images' => $backgroundImages,
      '#data' => [
        'url_scan_status' => $data['url_scan_status'] ?? 0,
        'scan_status' => $data['scan_status'] ?? 0,
        'total_selected_pages' => $data['total_selected_pages'] ?? 0,
        'total_scan_pages' => $data['total_scan_pages'] ?? 0,
        'last_scan' => $data['last_scan'] ?? '',
        'success_percentage' => $data['success_percentage'] ?? 0,
        'scan_violation_total' => $data['scan_violation_total'] ?? 0,
        'total_fail_sum' => $data['total_fail_sum'] ?? 0,
        'sitemap_pages' => $data['total_pages'] ?? 0,
        'dashboard_link' => $data['dashboard_link'] ?? '#',
        'domain' => $data['domain'] ?? '',
        'fav_icon' => $data['fav_icon'] ?? '',
        'total_last_scan_pages' => $data['total_last_scan_pages'] ?? 0,
        'last_url_scan' => $data['last_url_scan'] ?? '',
        'next_scan_date' => $data['next_scan_date'] ?? '',
        'total_violations' => $data['total_violations'] ?? 0,
        'website_id' => $data['website_id'] ?? '',
        'scan_success_total' => $data['scan_success_total'] ?? 0,
        'total_success_sum' => $data['total_success_sum'] ?? 0,
        'fail_percentage' => $data['fail_percentage'] ?? 0,
        'total_scan_monthly_pages' => $data['total_scan_monthly_pages'] ?? 0,
        'total_scanning_process' => $data['total_scanning_process'] ?? 0,
        'is_pause_scanning' => $data['is_pause_scanning'] ?? 0,
      ],

      '#sitemap_pages' => $data['total_pages'] ?? 0,
      '#scan_score_percentage' => $display_score_value,
      '#total_violations' => $data['total_fail_sum'] ?? 0,
      '#last_scan_date' => !empty($data['last_scan']) ? $data['last_scan'] : 'Not Scanned',
      '#total_scan_pages' => $data['total_scan_pages'] ?? 0,
      '#plan_name' => $data['package_name'] ?? '',
      '#plan_pages' => $data['page_views'] ?? 0,
      '#renewal_date' => $plan_display_date,
      '#cancel_subscription_link' => $data['dashboard_link'] ?? '',
      '#plan_action_link' => $data['plan_action_link'] ?? ($data['dashboard_link'] ?? '#'),
      '#plans' => $data['plans'] ?? [],
      '#billing_interval' => $data['subscr_interval'] ?? 'M',
      '#accessibility_score' => $display_score_value,
      '#accessibility_status_class' => $statusClass,
      '#accessibility_status_text' => $statusText,
      '#total_scan_pages_report' => $data['total_scan_pages'] ?? 0,
      '#total_pages_report' => $data['total_pages'] ?? 0,
      '#plan_status' => $data['plan_status'] ?? $plan_status,
      '#renewal_text' => $data['renewal_text'] ?? $renewal_text,
      '#button_text' => $data['button_text'] ?? $button_text,
      '#is_expired' => $is_expired,
      '#final_price' => $data['final_price'] ?? 0,
      '#payment_status' => $data['payment_status'] ?? '',
      '#is_trial_period' => $data['is_trial_period'] ?? 0,
      '#scan_status_text' => $scan_status_text,
      '#scan_status_icon' => $scan_status_icon,
      '#scanning_paused' => $scanning_paused,
      '#scan_pause_reason' => $scan_pause_reason,

      // FIXED: WCAG data with proper fallbacks and debugging.
      '#wcag_failed' => $data['scan_details']['total_fail'] ??
      $data['scan_details_with_remediation']['total_fail'] ??
      $data['scan_details_without_remediation']['total_fail'] ?? 0,

      '#wcag_passed' => $data['scan_details']['total_success'] ??
      $data['scan_details_with_remediation']['total_success'] ??
      $data['scan_details_without_remediation']['total_success'] ?? 0,

      '#wcag_na' => $data['scan_details']['severity_counts']['Not_Applicable'] ??
      $data['scan_details_with_remediation']['severity_counts']['Not_Applicable'] ??
      $data['scan_details_without_remediation']['severity_counts']['Not_Applicable'] ?? 0,

      '#level_a_violations' => $data['scan_details']['criteria_counts']['A'] ??
      $data['scan_details_with_remediation']['criteria_counts']['A'] ??
      $data['scan_details_without_remediation']['criteria_counts']['A'] ?? 0,

      '#level_aa_violations' => $data['scan_details']['criteria_counts']['AA'] ??
      $data['scan_details_with_remediation']['criteria_counts']['AA'] ??
      $data['scan_details_without_remediation']['criteria_counts']['AA'] ?? 0,

      '#level_aaa_violations' => $data['scan_details']['criteria_counts']['AAA'] ??
      $data['scan_details_with_remediation']['criteria_counts']['AAA'] ??
      $data['scan_details_without_remediation']['criteria_counts']['AAA'] ?? 0,

      '#violation_link' => $data['violation_link'] ?? '#',
      '#attached' => ['library' => ['skynetaccessibility_scanner/dashboard']],
    ];

    // Disable all caching for this page.
    $build['#cache'] = [
      'max-age' => 0,
      'contexts' => ['url.path', 'url.query_args'],
      'tags' => ['skynetaccessibility_scanner:scan_data'],
    ];

    return $build;
  }

}
