<?php

namespace Drupal\sleepy_cron;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\CronInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Site\Settings;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Decorates the core cron service.
 */
class SleepyCron implements CronInterface {

  use StringTranslationTrait;

  /**
   * SleepyCron constructor.
   *
   * @param \Drupal\Core\CronInterface $innerCron
   *   Decorated cron service.
   * @param \Drupal\sleepy_cron\SleepyCronStatus $status
   *   The sleepy cron status service.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerChannelFactory
   *   The logger channel factory service.
   */
  public function __construct(
    protected CronInterface $innerCron,
    private readonly SleepyCronStatus $status,
    private readonly TimeInterface $time,
    private readonly LoggerChannelFactoryInterface $loggerChannelFactory,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function run(): bool {
    $logger = $this->loggerChannelFactory->get('sleepy_cron');

    if ($this->status->sleepyCronFeatureIsEnabled()
      && $this->status->cronIsAsleep()) {
      $logger->warning($this->t('Cron is asleep so cron jobs will not run.'));
      // We do not run cron jobs when cron is already asleep.
      return TRUE;
    }

    // If there has been much time* since the latest website usage, it's time to
    // get the cron to sleep.
    // * "much time" defaults to 6 hours.
    $now = $this->time->getRequestTime();
    $time_before_sleeping = Settings::get('sleepy_cron.time_before_sleeping', 21600);
    $get_to_sleep_at = $this->status->getLatestRecordedUsageTime() + $time_before_sleeping;
    if ($this->status->sleepyCronFeatureIsEnabled()
      && $now >= $get_to_sleep_at) {
      $this->status->getCronToSleep();
      $logger->info($this->t('Cron is now asleep.'));
      // We do not run cron jobs when cron is asleep.
      return TRUE;
    }

    // Normal cron execution.
    return $this->innerCron->run();
  }

}
