<?php

namespace Drupal\sleepy_cron;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\CronInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Site\Settings;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Event subscriber for sleepy_cron.
 */
class SleepyCronEventSubscriber implements EventSubscriberInterface {

  use StringTranslationTrait;

  /**
   * SleepyCronEventSubscriber constructor.
   *
   * @param \Drupal\sleepy_cron\SleepyCronStatus $status
   *   The sleepy cron status service.
   * @param \Drupal\Core\CronInterface $cron
   *   The cron service.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current account.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Routing\RouteMatchInterface $currentRouteMatch
   *   The current route match.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   */
  public function __construct(
    private readonly SleepyCronStatus $status,
    private readonly CronInterface $cron,
    private readonly AccountProxyInterface $currentUser,
    private readonly MessengerInterface $messenger,
    private readonly RouteMatchInterface $currentRouteMatch,
    private readonly TimeInterface $time,
  ) {}

  /**
   * Wake up the cron if it is asleep.
   */
  public function wakeCronUpIfAsleep(): void {
    if (!$this->status->sleepyCronFeatureIsEnabled()) {
      return;
    }

    if ($this->status->cronIsAsleep()) {
      // If cron is launched through the cron launch URL, we do not want to wake
      // cron up. This is not a "real usage" of the website.
      if (!$this->requestIsRealUsage()) {
        return;
      }

      // This is the first request for a long time.
      // The website is used again, so crons should run normally.
      $this->status->wakeCronUp();

      // Some cron jobs should have run but could not because cron was asleep.
      // So we execute cron once now. If there are queues, they may not have run
      // entirely but that's way better than nothing.
      $this->cron->run();

      // We tell about the risk of late crons.
      if ($this->currentUser->hasPermission('see cron wakeup warning')) {
        $this->messenger->addWarning($this->t('You are the first to
 request this website for a long time. So that to preserve server
 resources, cron had been temporarily disabled. It just ran again, but you
 may expect strange behaviors because some queues handled on cron (such as
 content indexation for search) could still have to be fully processed.'));
      }
    }
  }

  /**
   * Keep the cron awake.
   */
  public function keepCronAwake(): void {
    if (!$this->status->sleepyCronFeatureIsEnabled()) {
      return;
    }

    // If cron is launched through the cron launch URL, we do not want to keep
    // cron awake. This is not a "real usage" of the website.
    if (!$this->requestIsRealUsage()) {
      return;
    }

    // We do not record every request for performance reasons.
    // Using session write interval seems like a good trade-off.
    $now = $this->time->getRequestTime();
    if ($this->status->getLatestRecordedUsageTime() < $now - Settings::get('session_write_interval', 180)) {
      $this->status->recordUsageTime($now);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events[KernelEvents::REQUEST][] = ['wakeCronUpIfAsleep'];
    $events[KernelEvents::TERMINATE][] = ['keepCronAwake'];
    return $events;
  }

  /**
   * Check if the request corresponds to a real user browsing the website.
   *
   * @return bool
   *   TRUE if the request is from a real user.
   */
  private function requestIsRealUsage(): bool {
    return !in_array($this->currentRouteMatch->getRouteName(), [
      // The route to launch the cron from outside:
      'system.cron',
      // The admin button to launch the cron send a POST request to:
      'system.cron_settings',
      // No route name, it's probably a drush call.
      NULL,
    ]);
  }

}
