<?php

namespace Drupal\sleepy_cron;

use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Site\Settings;
use Drupal\Core\State\StateInterface;
use Drupal\Core\File\FileExists;

/**
 * Service that manages the cron state.
 *
 * This stores the cron's "state" (awake or asleep) and the latest time at which
 * the website was used.
 *
 * State is stored in a file so that it can easily be checked by an external
 * script. This allows this external script to not even execute the
 * cron-triggering command if it's asleep, thus preventing a Drush/Drupal
 * bootstrap, for better performance.
 * Anyway, if the cron-triggering command is executed while cron is asleep, it
 * will do nothing but a bootstrap and a check. No jobs or queues execution.
 */
class SleepyCronStatus {

  const string FILE_NAME = 'cron_is_asleep.txt';

  const string FILE_CONTENT = "This file created by the Sleepy cron Drupal module
stores the fact that the cron system is currently asleep because the website
has not been used for a long time. It's checked by the module to prevent cron
execution, but can be checked by an external script even before triggering a
cron execution, for better performance.";

  const string STATE_NAME = 'sleepy_cron_latest_usage';

  /**
   * SleepyCronStatus constructor.
   *
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The file system service.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   */
  public function __construct(
    private readonly FileSystemInterface $fileSystem,
    private readonly StateInterface $state,
  ) {}

  /**
   * Check if the cron is asleep.
   *
   * @return bool
   *   TRUE if the cron is asleep.
   */
  public function cronIsAsleep(): bool {
    $mask = '/^' . preg_quote(self::FILE_NAME) . '$/';
    return !empty($this->fileSystem->scanDirectory($this->getCronStateFileDir(), $mask, ['recurse' => FALSE]));
  }

  /**
   * Make the cron asleep.
   */
  public function getCronToSleep(): void {
    $this->fileSystem->saveData(self::FILE_CONTENT, $this->getCronStateFileUri(), FileExists::Replace);
  }

  /**
   * Wake up the cron.
   */
  public function wakeCronUp(): void {
    $this->fileSystem->unlink($this->getCronStateFileUri());
  }

  /**
   * Get the latest time the cron was used.
   *
   * @return int
   *   The latest time the cron was used.
   */
  public function getLatestRecordedUsageTime(): int {
    return $this->state->get(self::STATE_NAME, 0);
  }

  /**
   * Set the latest time the cron was used.
   *
   * @param int $usage_time
   *   The latest time the cron was used.
   */
  public function recordUsageTime(int $usage_time): void {
    $this->state->set(self::STATE_NAME, $usage_time);
  }

  /**
   * Allows to easily disable the feature on some environments such as prod.
   *
   * @return bool
   *   TRUE if sleepy_cron is enabled.
   */
  public function sleepyCronFeatureIsEnabled() : bool {
    return Settings::get('sleepy_cron.enabled', TRUE);
  }

  /**
   * Get the URI of the file containing the cron state.
   *
   * @return string
   *   URI of the file containing the cron state.
   */
  private function getCronStateFileUri(): string {
    return $this->getCronStateFileDir() . '/' . self::FILE_NAME;
  }

  /**
   * Get the path of the folder containing the cron state file.
   *
   * @return string
   *   Path of the folder containing the cron state file.
   */
  private function getCronStateFileDir(): string {
    /*
     * We default to the public dir because:
     * - it's writable,
     * - a private dir might not exist,
     * - it's not secret data,
     * - and there's a different one for each different site in multisite mode.
     */
    return Settings::get('sleepy_cron.cron_state_file_dir', 'public://');
  }

}
