<?php

namespace Drupal\slick_pro_customizer\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Provides a "Slick Pro Customizer" block.
 *
 * @Block(
 *   id = "slick_pro_customizer_block",
 *   admin_label = @Translation("Slick Pro Customizer Block")
 * )
 */
class SlickProBlock extends BlockBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'elements' => [],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $elements = $form_state->get('elements') ?? $this->configuration['elements'] ?? [];
    $form_state->set('elements', $elements);

    $form['elements'] = [
      '#type' => 'table',
      '#header' => [],
      '#empty' => $this->t('No elements added yet.'),
      '#tree' => TRUE,
    ];

    foreach ($elements as $delta => $element) {
      // Primary Options.
      $form['elements'][$delta]['primary_options'] = [
        '#type' => 'details',
        '#title' => $this->t('Primary Options'),
        '#open' => TRUE,
      ];

      $form['elements'][$delta]['primary_options'] += [
        'selector' => [
          '#type' => 'textfield',
          '#title' => $this->t('Selector / Primary'),
          '#default_value' => $element['selector'] ?? '',
          '#required' => TRUE,
        ],
        'slides_desktop' => [
          '#type' => 'number',
          '#title' => $this->t('Slides to Show (Desktop)'),
          '#default_value' => $element['slides_desktop'] ?? 3,
          '#min' => 1,
          '#max' => 10,
        ],
        'slides_mobile' => [
          '#type' => 'number',
          '#title' => $this->t('Slides to Show (Mobile)'),
          '#default_value' => $element['slides_mobile'] ?? 1,
          '#min' => 1,
          '#max' => 10,
        ],
        'autoplay' => [
          '#type' => 'checkbox',
          '#title' => $this->t('Autoplay'),
          '#default_value' => $element['autoplay'] ?? FALSE,
        ],
        'autoplay_speed' => [
          '#type' => 'number',
          '#title' => $this->t('Autoplay Speed (ms)'),
          '#default_value' => $element['autoplay_speed'] ?? 3000,
          '#min' => 100,
        ],
        'speed' => [
          '#type' => 'number',
          '#title' => $this->t('Transition Speed (ms)'),
          '#default_value' => $element['speed'] ?? 500,
          '#min' => 100,
        ],
      ];

      // Secondary Options.
      $form['elements'][$delta]['secondary_options'] = [
        '#type' => 'details',
        '#title' => $this->t('Secondary Options'),
        '#open' => FALSE,
        'dots' => [
          '#type' => 'checkbox',
          '#title' => $this->t('Dots'),
          '#default_value' => $element['dots'] ?? TRUE,
        ],
        'arrows' => [
          '#type' => 'checkbox',
          '#title' => $this->t('Arrows'),
          '#default_value' => $element['arrows'] ?? TRUE,
        ],
        'infinite' => [
          '#type' => 'checkbox',
          '#title' => $this->t('Infinite'),
          '#default_value' => $element['infinite'] ?? TRUE,
        ],
        'custom_prev_arrow' => [
          '#type' => 'textfield',
          '#title' => $this->t('Custom Prev Arrow'),
          '#default_value' => $element['custom_prev_arrow'] ?? '',
          '#states' => [
            'visible' => [
              ':input[name="elements[' . $delta . '][secondary_options][arrows]"]' => ['checked' => TRUE],
            ],
          ],
        ],
        'custom_next_arrow' => [
          '#type' => 'textfield',
          '#title' => $this->t('Custom Next Arrow'),
          '#default_value' => $element['custom_next_arrow'] ?? '',
          '#states' => [
            'visible' => [
              ':input[name="elements[' . $delta . '][secondary_options][arrows]"]' => ['checked' => TRUE],
            ],
          ],
        ],
        'center_mode' => [
          '#type' => 'checkbox',
          '#title' => $this->t('Center Mode'),
          '#default_value' => $element['center_mode'] ?? FALSE,
        ],
        'center_padding_desktop' => [
          '#type' => 'textfield',
          '#title' => $this->t('Center Padding (Desktop)'),
          '#default_value' => $element['center_padding_desktop'] ?? '60px',
          '#description' => $this->t('CSS padding value (e.g., 60px, 10%).'),
          '#states' => [
            'visible' => [
              ':input[name="elements[' . $delta . '][secondary_options][center_mode]"]' => ['checked' => TRUE],
            ],
          ],
        ],
        'center_padding_mobile' => [
          '#type' => 'textfield',
          '#title' => $this->t('Center Padding (Mobile)'),
          '#default_value' => $element['center_padding_mobile'] ?? '30px',
          '#description' => $this->t('CSS padding value (e.g., 30px, 5%).'),
          '#states' => [
            'visible' => [
              ':input[name="elements[' . $delta . '][secondary_options][center_mode]"]' => ['checked' => TRUE],
            ],
          ],
        ],
      ];

      $form['elements'][$delta]['remove'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Remove'),
      ];
    }

    $form['add_element'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add Element'),
      '#submit' => [[static::class, 'addElement']],
      '#limit_validation_errors' => [],
    ];

    return $form;
  }

  /**
   * Adds a new element row to the form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public static function addElement(array &$form, FormStateInterface $form_state): void {
    $elements = $form_state->get('elements') ?? [];
    $elements[] = [
      'selector' => '',
      'slides_desktop' => 3,
      'slides_mobile' => 1,
      'autoplay' => FALSE,
      'autoplay_speed' => 3000,
      'speed' => 500,
      'dots' => TRUE,
      'arrows' => TRUE,
      'infinite' => TRUE,
      'custom_prev_arrow' => '',
      'custom_next_arrow' => '',
      'remove' => FALSE,
      'center_mode' => FALSE,
      'center_padding_desktop' => '60px',
      'center_padding_mobile' => '30px',
    ];
    $form_state->set('elements', $elements);
    $form_state->setRebuild(TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state): void {
    $elements = $form_state->getValue('elements') ?? [];

    foreach ($elements as $key => $el) {
      if (!empty($el['remove'])) {
        unset($elements[$key]);
        continue;
      }

      $elements[$key] = [
        'selector' => (string) ($el['primary_options']['selector'] ?? ''),
        'slides_desktop' => (int) ($el['primary_options']['slides_desktop'] ?? 3),
        'slides_mobile' => (int) ($el['primary_options']['slides_mobile'] ?? 1),
        'autoplay' => !empty($el['primary_options']['autoplay']),
        'autoplay_speed' => (int) ($el['primary_options']['autoplay_speed'] ?? 3000),
        'speed' => (int) ($el['primary_options']['speed'] ?? 500),
        'dots' => !empty($el['secondary_options']['dots']),
        'arrows' => !empty($el['secondary_options']['arrows']),
        'infinite' => !empty($el['secondary_options']['infinite']),
        'custom_prev_arrow' => (string) ($el['secondary_options']['custom_prev_arrow'] ?? ''),
        'custom_next_arrow' => (string) ($el['secondary_options']['custom_next_arrow'] ?? ''),
        'center_mode' => !empty($el['secondary_options']['center_mode']),
        'center_padding_desktop' => (string) ($el['secondary_options']['center_padding_desktop'] ?? '60px'),
        'center_padding_mobile' => (string) ($el['secondary_options']['center_padding_mobile'] ?? '30px'),
      ];
    }

    $this->configuration['elements'] = array_values($elements);
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    return [
      '#markup' => '',
      '#attached' => [
        'library' => ['slick_pro_customizer/slick_customizer'],
        'drupalSettings' => [
          'slickProCustomizer' => $this->configuration['elements'],
        ],
      ],
    ];
  }

}
