<?php

namespace Drupal\slider_gallery\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\file\FileInterface;
use Drupal\image\Entity\ImageStyle;
use Drupal\media\Entity\Media;
use Drupal\file\Entity\File;

/**
 * Plugin implementation of the 'image_gallery_formatter'.
 *
 * @FieldFormatter(
 *   id = "image_gallery_formatter",
 *   label = @Translation("Galería de imágenes (Fancybox)"),
 *   field_types = {
 *     "image",
 *     "entity_reference"
 *   }
 * )
 */
class ImageGalleryFormatter extends FormatterBase
{

  public static function defaultSettings(): array
  {
    return array_merge(parent::defaultSettings(), [
      'max_images' => 5,
      'image_style' => '',
    ]);
  }

  public function settingsForm(array $form, FormStateInterface $form_state): array
  {
    $elements = parent::settingsForm($form, $form_state);

    $styles = ImageStyle::loadMultiple();
    $options = ['' => $this->t('- Sin estilo -')];
    foreach ($styles as $id => $style) {
      $options[$id] = $style->label();
    }

    $elements['max_images'] = [
      '#type' => 'number',
      '#title' => $this->t('Máximo de imágenes visibles'),
      '#default_value' => $this->getSetting('max_images'),
      '#min' => 1,
    ];

    $elements['image_style'] = [
      '#type' => 'select',
      '#title' => $this->t('Estilo de imagen'),
      '#options' => $options,
      '#default_value' => $this->getSetting('image_style'),
    ];

    return $elements;
  }

  public function viewElements(FieldItemListInterface $items, $langcode): array
  {
    $max = (int)$this->getSetting('max_images');
    $style = $this->getSetting('image_style');
    $gallery_id = 'gallery-' . uniqid();

    $images = [];
    $hidden = [];
    $index = 0;

    foreach ($items as $item) {
      $file = NULL;
      $alt = '';
      $title = '';

      if ($item->entity instanceof FileInterface) {
        $file = $item->entity;
        $alt = $item->alt ?? '';
        $title = $item->title ?? '';
      } elseif ($item->entity instanceof Media && $item->entity->bundle() === 'image') {
        $media_image = $item->entity->get('field_media_image')->first();
        if ($media_image) {
          $file = File::load($media_image->target_id);
          $alt = $media_image->alt;
          $title = $media_image->title;
        }
      }

      if (!$file) {
        continue;
      }

      $url = $style && ImageStyle::load($style)
        ? ImageStyle::load($style)->buildUrl($file->getFileUri())
        : \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());

      $data = [
        'url' => $url,
        'alt' => $alt,
        'title' => $title,
        'gallery_id' => $gallery_id,
      ];

      if ($index < $max) {
        $images[] = $data;
      } else {
        $hidden[] = $data;
      }

      $index++;
    }

    return [[
      '#theme' => 'image_gallery',
      '#images' => $images,
      '#hidden_images' => $hidden,
      '#gallery_id' => $gallery_id,
      '#attached' => [
        'library' => [
          'slider_gallery/fancybox',
        ],
      ],
    ]];
  }

}
