<?php

namespace Drupal\slider_gallery\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\Annotation\FieldFormatter;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Image\ImageFactory;
use Drupal\file\Entity\File;
use Drupal\file\FileInterface;
use Drupal\image\Entity\ImageStyle;
use Drupal\media\Entity\Media;
use Drupal\node\NodeInterface;

/**
 * Plugin implementation of the 'image_slider_formatter' formatter.
 *
 * @FieldFormatter(
 *   id = "image_slider_formatter",
 *   label = @Translation("Slider de Imágenes"),
 *   field_types = {
 *     "image",
 *     "entity_reference"
 *   }
 * )
 */
class ImageSliderFormatter extends FormatterBase
{

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings()
  {
    return array_merge(parent::defaultSettings(), [
      'items_to_show' => 5,  // Valor predeterminado: mostrar 5 imágenes
      'link_behavior' => 'none',
      'image_style' => '',
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state)
  {
    $elements = parent::settingsForm($form, $form_state);

    $image_styles = ImageStyle::loadMultiple();
    $options = ['' => $this->t('- Sin estilo -')];
    foreach ($image_styles as $style_id => $style) {
      $options[$style_id] = $style->label();
    }

    $elements['image_style'] = [
      '#type' => 'select',
      '#title' => $this->t('Estilo de imagen'),
      '#default_value' => $this->getSetting('image_style'),
      '#options' => $options,
      '#description' => $this->t('Selecciona un estilo de imagen para aplicar al slider.'),
    ];

    // Campo para seleccionar la cantidad de imágenes a mostrar
    $elements['items_to_show'] = [
      '#type' => 'number',
      '#title' => $this->t('Cantidad de imágenes a mostrar'),
      '#default_value' => $this->getSetting('items_to_show'),
      '#min' => 1,
      '#description' => $this->t('Define la cantidad de imágenes que se mostrarán en el slider.'),
    ];

    $elements['link_behavior'] = [
      '#type' => 'select',
      '#title' => $this->t('Link image to'),
      '#default_value' => $this->getSetting('link_behavior'),
      '#options' => [
        'none' => $this->t('Nothing'),
        'content' => $this->t('Content'),
        'file' => $this->t('File'),
      ],
      '#description' => $this->t('Define the link behavior for the slider.'),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary()
  {
    $summary = parent::settingsSummary();
    $summary[] = $this->t('Mostrar @items imágenes', ['@items' => $this->getSetting('items_to_show')]);
    $link_behavior = $this->getSetting('link_behavior');

    $summary[] = match ($link_behavior) {
      'file' => $this->t('Las imágenes enlazan al archivo.'),
      'content' => $this->t('Las imágenes enlazan al contenido.'),
      default => $this->t('Las imágenes no tienen enlace.'),
    };

    $image_style = $this->getSetting('image_style');
    $summary[] = $image_style
      ? $this->t('Estilo de imagen: @style', ['@style' => $image_style])
      : $this->t('Sin estilo de imagen.');

    return $summary;
  }

  public function viewElements(FieldItemListInterface $items, $langcode)
  {
    $images = [];
    $allimages = [];
    $extra_images = [];
    $items_to_show = $this->getSetting('items_to_show');
    $link_behavior = $this->getSetting('link_behavior');
    $image_style = $this->getSetting('image_style');

    $thumb_style_id = 'medium';

    foreach ($items as $index => $item) {
      /*if ($index >= $items_to_show) {
        break;  // Detener el ciclo si ya hemos mostrado la cantidad configurada de imágenes
      }*/

      $target_entity = $item->entity;
      $file = NULL;

      // Si es un archivo de imagen directo.
      if ($target_entity instanceof FileInterface) {
        $file = $target_entity;
        $alt = $item->get('alt')->getString();
        $title = $item->get('title')->getString();
      } // Si es un Media del tipo image.
      elseif ($target_entity instanceof Media && $target_entity->bundle() === 'image') {
        // Ajusta 'field_media_image' si tu campo tiene otro nombre.
        if ($target_entity->hasField('field_media_image') && !$target_entity->get('field_media_image')->isEmpty()) {
          $image_item_media = $target_entity->get('field_media_image')->first();
          if ($image_item_media) {

            // 2) Carga el File de forma segura (sin acceder a ->entity directamente).
            $fid = (int) $image_item_media->get('target_id')->getString();
            /** @var \Drupal\file\FileInterface|null $file */
            $file = $fid ? File::load($fid) : NULL;

            // 3) Lee alt y title desde typed data (sin tocar props protegidas).
            $alt   = $image_item_media->get('alt')->getString();
            $title = $image_item_media->get('title')->getString();
          }
        }
      }
      //file = $item->entity;

      if ($file) {
        // $imageUrl = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());

        if ($image_style && ImageStyle::load($image_style)) {
          $imageUrl = ImageStyle::load($image_style)->buildUrl($file->getFileUri());
        } else {
          $imageUrl = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
        }

        // URL de miniatura 100x100 (estilo "thumbnail")
        if (ImageStyle::load($thumb_style_id)) {
          $thumbUrl = ImageStyle::load($thumb_style_id)->buildUrl($file->getFileUri());
        } else {
          // Fallback por si alguien borró el estilo "thumbnail".
          $thumbUrl = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
        }


        $linkUrl = null;

        switch ($link_behavior) {
          case 'file':
            $linkUrl = $imageUrl;
            break;
          case 'content':
            $parent = $item->getEntity();
            if ($parent instanceof NodeInterface) {
              $linkUrl = $parent->toUrl()->toString();
            }
            break;
          case 'none':
            $linkUrl = NULL;
            break;
        }

        $image_item = [
          'image_url' => $imageUrl,
          'thumb_url' => $thumbUrl,
          'link_url' => $linkUrl,
          'alt' => $alt,
          'title' => $title,
        ];

        $allimages[] = $image_item;

        if ($index < $items_to_show) {
          $images[] = $image_item;
        } else {
          $extra_images[] = $image_item;
        }
      }
    }

    // Construimos el render array para usar la plantilla Twig.
    $elements = [
      '#theme' => 'image_slider',
      '#images' => $images,
      '#extra_images' => $extra_images,
      '#allimages' => $allimages,
      // Usamos un identificador único para diferenciar instancias (por ejemplo, el field name y delta).
      '#slider_id' => $this->fieldDefinition->getName() . '-' . uniqid(),
      '#attached' => [
        'library' => [
          'slider_gallery/slider',
          'slider_gallery/modal',
        ],
      ],
    ];

    return [$elements];
  }

}
