<?php

declare(strict_types=1);

namespace Drupal\slimmer\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\slimmer\Log\SlimmerLogLevels;

/**
 * Configure Slimmer module settings.
 */
final class SettingForm extends ConfigFormBase {

  /**
   * Config settings.
   *
   * @var string
   */
  const SETTINGS = 'slimmer.settings';

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'slimmer_admin_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [
      static::SETTINGS,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config(static::SETTINGS);

    $form['slimmer_project'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Project name'),
      '#default_value' => $config->get('slimmer_project'),
    ];

    $form['slimmer_status'] = [
      '#type' => 'radios',
      '#title' => $this->t('Status'),
      '#description' => $this->t('Enable or disable the Slimmer entirely.'),
      '#default_value' => $config->get('slimmer_status') ?? 0,
      '#options' => [
        0 => $this->t('Disabled'),
        1 => $this->t('Enabled'),
      ],
    ];

    $form['slimmer_autolog'] = [
      '#type' => 'radios',
      '#title' => $this->t('Autologger enabled'),
      '#description' => $this->t('Automatically log messages to Slimmer based on PSR logging'),
      '#default_value' => $config->get('slimmer_autolog'),
      '#options' => [
        0 => $this->t('Disabled'),
        1 => $this->t('Enabled'),
      ],
    ];

    $form['slimmer_endpoint'] = [
      '#type' => 'textfield',
      '#title' => $this->t('The endpoint to post requests to'),
      '#default_value' => $config->get('slimmer_endpoint'),
    ];

    $form['slimmer_log_level'] = [
      '#type' => 'select',
      '#title' => $this->t('Minimum log level to display'),
      '#default_value' => $config->get('slimmer_log_level') ?? SlimmerLogLevels::DEFAULT_LEVEL,
      '#options' => array_map(
        fn($label) => $this->t($label),
        SlimmerLogLevels::getLogLevelLabels()
      ),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config(static::SETTINGS)
      ->set('slimmer_project', $form_state->getValue('slimmer_project'))
      ->set('slimmer_status', $form_state->getValue('slimmer_status'))
      ->set('slimmer_autolog', $form_state->getValue('slimmer_autolog'))
      ->set('slimmer_endpoint', $form_state->getValue('slimmer_endpoint'))
      ->set('slimmer_log_level', $form_state->getValue('slimmer_log_level'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
