<?php

namespace Drupal\slimmer\Service;

use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\slimmer\Log\SlimmerLogLevels;
use GuzzleHttp\ClientInterface;
use Drupal\Core\Logger\RfcLoggerTrait;
use GuzzleHttp\Exception\GuzzleException;
use Psr\Log\LoggerInterface;
use Psr\Log\LogLevel;

/**
 * Handles logging for the Slimmer project.
 */
class LogService implements LoggerInterface {

  use RfcLoggerTrait;

  private const ROOT_NAME = 'ROOT';

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * Constructor.
   *
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   The HTTP client.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger service.
   */
  public function __construct(ClientInterface $httpClient, LoggerChannelInterface $logger) {
    $this->httpClient = $httpClient;
    $this->logger = $logger;
  }

  /**
   * Logs a message with the custom 'slimmer' log level.
   *
   * @param string $message
   *   The log message.
   */
  public function logSlimmerMessage(string $message): void {
    $this->logger->info($message);
  }

  /**
   * Gets the browser name from the user agent string.
   *
   * @return string
   *   The browser name.
   */
  private function getBrowserName(): string {
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';

    return match (true) {
      str_contains($userAgent, 'Firefox') => 'Mozilla Firefox',
      str_contains($userAgent, 'Chrome') => 'Google Chrome',
      str_contains($userAgent, 'Safari') => 'Safari',
      str_contains($userAgent, 'Opera'), str_contains($userAgent, 'OPR') => 'Opera',
      str_contains($userAgent, 'Edge') => 'Microsoft Edge',
      str_contains($userAgent, 'MSIE'), str_contains($userAgent, 'Trident') => 'Internet Explorer',
      default => self::ROOT_NAME,
    };
  }

  /**
   * Gets the current request URL, or returns 'ROOT' if it contains 'default'.
   *
   * @return string
   *   The request URL or 'ROOT' if matched.
   */
  private function getRequestSource(): string {
    $server_name = $_SERVER['SERVER_NAME'] ?? '';
    $request_uri = $_SERVER['REQUEST_URI'] ?? '';
    $full_url = $server_name . $request_uri;

    if (str_contains($full_url, 'default')) {
      return self::ROOT_NAME;
    }

    // Limit the string to 100 chars.
    return mb_strlen($full_url) > 200
      ? mb_substr($full_url, 0, 200) . '..'
      : $full_url;
  }

  /**
   * Maps PSR-3 level string to internal numeric severity using SlimmerLogLevels.
   *
   * @param string $level
   *   PSR-3 log level string.
   *
   * @return string Numeric severity (lower is less severe).
   *   Numeric severity (lower is less severe).
   */
  private function mapLogLevel(string $level): string {
    return SlimmerLogLevels::getLevelName($level) ?? SlimmerLogLevels::getLevelName('3');
  }

  /**
   * Builds the array for the external Slimmer log endpoint.
   */
  private function createPostArray(string $level, string $message): array|false {
    $config = \Drupal::config('slimmer.settings');

    if (
      !$message ||
      !$config->get('slimmer_project')
    ) {
      return FALSE;
    }

    return [
      'project' => $config->get('slimmer_project'),
      'timestamp' => time(),
      'browser' => $this->getBrowserName(),
      'level' => $this->mapLogLevel($level) ,
      'data' => [
        'message' => $message,
      ],
      'user' => \Drupal::currentUser()->id(),
      'location' => $this->getRequestSource(),
    ];
  }

  /**
   * PSR-3 log method implementation.
   *
   *  - Will be called by LogEventSubscriber automatically.
   *
   * @throws GuzzleException
   */
  public function log($level, string|\Stringable $message, array $context = []): void {
    // Only handle logs sent to the 'slimmer' channel.
    if (($context['channel'] ?? null) !== 'slimmer') {
      return;
    }

    // Check for log level severity.
    $config = \Drupal::config('slimmer.settings');
    $minLevel = $config->get('slimmer_log_level') ?? LogLevel::DEBUG;

    $minSeverity = SlimmerLogLevels::getSeverity($minLevel);

    if ($minSeverity === null) {
      \Drupal::logger('slimmer')->warning('Unknown minimum log level severity from config: @level', ['@level' => $minLevel]);
      return;
    }

    if ($level > $minSeverity) {
      return;
    }

    $this->postLog($level, $message);
  }

  /**
   * Slimmer log with no severity check.
   *
   *  - Can be called directly from custom code.
   *
   * @throws GuzzleException
   */
  public function slimmerLog(string $level, string $message): bool {
    return $this->postLog($level, $message);
  }

  /**
   * Post the log to the Slimmer endpoint.
   *
   * @param string $level
   * @param string $message
   * @return bool
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function postLog(string $level, string $message): bool {
    $config = \Drupal::config('slimmer.settings');

    // Slimmer is disabled, do not log.
    if ($config->getRawData()['slimmer_status'] === "0") {
      return FALSE;
    }

    // Autolog is disabled, do not log.
    if ($config->getRawData()['slimmer_autolog'] === "0") {
      return FALSE;
    }

    // Spam filter: prevent same message within 5 seconds.
    $state = \Drupal::state();

    $lastLog = $state->get('slimmer_last_log') ?? [
      'message' => '',
      'timestamp' => 0,
    ];

    $now = time();

    if (
      $message === $lastLog['message'] &&
      ($now - $lastLog['timestamp']) < 5
    ) {
      return FALSE;
    }

    $state->set('slimmer_last_log', [
      'message' => $message,
      'timestamp' => $now,
    ]);

    // 3. Create the post array and return early if invalid.
    $postArray = $this->createPostArray($level, $message);

    if (!$postArray) {
      return FALSE;
    }

    // 4. Send message to the Slimmer endpoint.
    try {
      $config = \Drupal::config('slimmer.settings');
      $this->httpClient->post($config->get('slimmer_endpoint'), [
        'headers' => ['Content-Type' => 'application/json'],
        'body' => json_encode($postArray),
      ]);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->logger->error('Cannot post log to dashboard from custom logger: ' . $e->getMessage());

      return FALSE;
    }
  }

}
