<?php

declare(strict_types=1);

namespace Drupal\slimmer\Logger;

use Drupal\slimmer\Log\SlimmerLogLevels;
use Psr\Log\AbstractLogger;
use Stringable;

/**
 * Logs all PSR-3 logging events for debugging and custom handling.
 */
final class LogEventSubscriber extends AbstractLogger {

  /**
   * Logs a message at the specified level.
   *
   * @param string|int $level
   *   The log level (string from LogLevel::* or numeric).
   * @param string|\Stringable $message
   *   The log message.
   * @param array $context
   *   Additional context for the message.
   */
  public function log($level, Stringable|string $message, array $context = []): void {
    if (is_int($level)) {
      $resolvedLevel = SlimmerLogLevels::getLevelName((string) $level);
      if ($resolvedLevel === null) {
        \Drupal::logger('slimmer')->warning('Invalid integer log level: @level', ['@level' => $level]);
        return;
      }
      $level = $resolvedLevel;
    }

    if (!is_string($level)) {
      \Drupal::logger('slimmer')->warning('Non-string log level passed: @level', ['@level' => print_r($level, TRUE)]);
      return;
    }

    // Format message.
    $message = slimmer_format_message($message, $context);

    // Log to Slimmer, using checks.
    \Drupal::service('slimmer.log')->log($level, $message);
  }

}
