<?php

namespace Drupal\slimmer\Log;

use Psr\Log\LogLevel;

/**
 * Defines log level constants and mappings for Slimmer.
 */
final class SlimmerLogLevels {

  /**
   * Maps PSR-3 log levels to severity integers.
   * Lower number = higher severity.
   */
  public const LEVEL_SEVERITY = [
    LogLevel::EMERGENCY => 0,
    LogLevel::ALERT     => 1,
    LogLevel::CRITICAL  => 2,
    LogLevel::ERROR     => 3,
    LogLevel::WARNING   => 4,
    LogLevel::NOTICE    => 5,
    LogLevel::INFO      => 6,
    LogLevel::DEBUG     => 7,
  ];

  /**
   * Gets the numeric severity for a PSR-3 level.
   */
  public static function getSeverity(string $level): ?int {
    return self::LEVEL_SEVERITY[$level] ?? NULL;
  }

  /**
   * Gets the PSR-3 level name from a numeric severity.
   */
  public static function getLevelName(string $severity): ?string {
    $flipped = array_flip(self::LEVEL_SEVERITY);
    return $flipped[$severity] ?? null;
  }

  /**
   * Returns all defined log levels with their severity.
   */
  public static function getAllLevels(): array {
    return self::LEVEL_SEVERITY;
  }

  /**
   * Returns human-readable labels for log levels.
   */
  public static function getLogLevelLabels(): array {
    return [
      LogLevel::EMERGENCY => 'Emergency',
      LogLevel::ALERT     => 'Alert',
      LogLevel::CRITICAL  => 'Critical',
      LogLevel::ERROR     => 'Error',
      LogLevel::WARNING   => 'Warning',
      LogLevel::NOTICE    => 'Notice',
      LogLevel::INFO      => 'Info',
      LogLevel::DEBUG     => 'Debug',
    ];
  }

  public const DEFAULT_LEVEL = LogLevel::DEBUG;
}
