<?php

declare(strict_types=1);

namespace Drupal\slimmer\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\slimmer\Log\SlimmerLogLevels;

/**
 * A test form for triggering logs at various levels.
 */
final class TestForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'slimmer_log_test_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['log_level'] = [
      '#type' => 'select',
      '#title' => $this->t('Log level'),
      '#options' => array_map(
        fn($label) => $this->t($label),
        SlimmerLogLevels::getLogLevelLabels()
      ),
      '#required' => TRUE,
    ];

    $form['message'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Log message'),
      '#default_value' => $this->t('Completed @test_variable of 10.'),
      '#description' => $this->t('Use @test_variable as a placeholder to include the test variable value.'),
      '#required' => TRUE,
    ];

    $form['test_variable'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Test variable value'),
      '#default_value' => '5',
    ];

    $form['actions'] = [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Log it'),
        '#button_type' => 'primary',
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $level = $form_state->getValue('log_level');
    $message = $form_state->getValue('message');
    $test_value = $form_state->getValue('test_variable');

    // Replace @test_variable in message.
    $message = strtr($message, [
      '@test_variable' => $test_value,
    ]);

    $logger = \Drupal::logger('slimmer');

    switch ($level) {
      case 'emergency':
        $logger->emergency($message);
        break;

      case 'alert':
        $logger->alert($message);
        break;

      case 'critical':
        $logger->critical($message);
        break;

      case 'error':
        $logger->error($message);
        break;

      case 'warning':
        $logger->warning($message);
        break;

      case 'notice':
        $logger->notice($message);
        break;

      case 'debug':
        $logger->debug($message);
        break;

      default:
        $logger->info($message);
        break;
    }

    $this->messenger()->addStatus($this->t('Logged message with Drupal log level: @level', ['@level' => $level]));
  }

}
