<?php

/**
 * @file
 * Provides Token integration for slots_paragraphs.
 */

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\paragraphs\ParagraphInterface;

/**
 * Implements hook_token_info().
 */
function slots_paragraphs_token_info() {
  $info = [];

  // Provide additional paragraph tokens.
  // @todo Remove this when https://www.drupal.org/project/paragraphs/issues/3067265 lands.
  $info['tokens']['paragraph']['delta'] = [
    'name' => t('Delta (position within parent field by bundle)'),
    'description' => t('Position of the paragraph among siblings of the same bundle within its parent field (1-based).'),
  ];

  // Expose a dynamic token that acts as a reference to the parent entity so
  // chained tokens like [paragraph:parent_entity:bundle] work automatically.
  $info['tokens']['paragraph']['parent_entity'] = [
    'name' => t('Parent entity'),
    'description' => t("Reference to the paragraph's parent entity. Allows chaining, e.g., [paragraph:parent_entity:entity-type], [paragraph:parent_entity:bundle], [paragraph:parent_entity:id], [paragraph:parent_entity:label]."),
    'dynamic' => TRUE,
  ];

  return $info;
}

/**
 * Implements hook_tokens().
 */
function slots_paragraphs_tokens($type, array $tokens, array $data, array $options = [], ?BubbleableMetadata $bubbleable_metadata = NULL) {
  if ($type !== 'paragraph') {
    return [];
  }

  $paragraph = $data['paragraph'] ?? NULL;
  if (!$paragraph instanceof ParagraphInterface) {
    return [];
  }

  $replacements = [];

  // Provide direct replacements we own.
  foreach ($tokens as $name => $original) {
    if ($name === 'delta') {
      $replacements[$original] = slots_paragraphs_compute_paragraph_delta($paragraph);
    }
    elseif ($name === 'parent_entity') {
      // If the top-level token [paragraph:parent_entity] is used without any
      // chain, fall back to the parent label for convenience.
      $parent = $paragraph->getParentEntity();
      if ($parent instanceof ContentEntityInterface) {
        if ($bubbleable_metadata) {
          $bubbleable_metadata->addCacheableDependency($parent);
        }
        $replacements[$original] = (string) $parent->label();
      }
      else {
        $replacements[$original] = '';
      }
    }
  }

  // Delegate chained parent_entity:* tokens to the parent entity type.
  $token_service = \Drupal::token();
  if ($parent_entity_tokens = $token_service->findWithPrefix($tokens, 'parent_entity')) {
    $parent = $paragraph->getParentEntity();
    if ($parent instanceof ContentEntityInterface) {
      if ($bubbleable_metadata) {
        $bubbleable_metadata->addCacheableDependency($parent);
      }
      $type = $parent->getEntityTypeId();
      $context = [$type => $parent];
      $replacements += $token_service->generate($type, $parent_entity_tokens, $context, $options, $bubbleable_metadata);
    }
    else {
      // If no parent entity, ensure chained tokens are cleared to empty.
      foreach ($parent_entity_tokens as $name => $original) {
        $replacements[$original] = '';
      }
    }
  }

  return $replacements;
}

/**
 * Computes the position of a paragraph within its parent field for its bundle.
 *
 * Mirrors the legacy getParagraphPosition() logic.
 */
function slots_paragraphs_compute_paragraph_delta(ParagraphInterface $paragraph) : string {
  $field_name = $paragraph->get('parent_field_name')->value ?? NULL;
  $parent = $paragraph->getParentEntity();

  if (!$field_name || !$parent || !isset($parent->{$field_name})) {
    return '';
  }

  $count = 0;
  $index_by_id = [];
  foreach ($parent->{$field_name}->referencedEntities() as $sibling) {
    if ($sibling->bundle() === $paragraph->bundle()) {
      $count++;
      $index_by_id[$sibling->id()] = $count;
    }
  }

  $id = $paragraph->id();
  return isset($index_by_id[$id]) ? (string) $index_by_id[$id] : '';
}
