<?php

namespace Drupal\slots_paragraphs\Plugin\paragraphs\Behavior;

use Drupal\Core\Entity\Display\EntityViewDisplayInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Utility\Token;
use Drupal\paragraphs\Attribute\ParagraphsBehavior;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\paragraphs\ParagraphInterface;
use Drupal\paragraphs\ParagraphsBehaviorBase;
use Drupal\slots\SlotsServiceInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a slot paragraphs behavior plugin.
 */
#[ParagraphsBehavior(
  id: 'slot',
  label: new TranslatableMarkup('Slot'),
  description: new TranslatableMarkup('Provides the option add a slot to the paragraph with a configurable slot id.'),
)]
class SlotBehavior extends ParagraphsBehaviorBase {

  /**
   * Constructs a SlotBehavior paragraphs behavior object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   * @param \Drupal\slots\SlotsServiceInterface $slotsService
   *   The slots' service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    EntityFieldManagerInterface $entity_field_manager,
    protected SlotsServiceInterface $slotsService,
    protected ModuleHandlerInterface $moduleHandler,
    protected Token $token,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $entity_field_manager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_field.manager'),
      $container->get('slots.service'),
      $container->get('module_handler'),
      $container->get('token'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return array_merge(parent::defaultConfiguration(), [
      'enable_pattern' => FALSE,
      'pattern' => 'slot_[paragraph:parent_entity:entity_type]_[paragraph:parent_entity:type]_[paragraph:delta]',
      'enable_cardinality' => FALSE,
      'cardinality' => 0,
      'label_pattern' => '[paragraph:parent_entity:label] — [paragraph:bundle] #[paragraph:delta]',
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['enable_pattern'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable default values (token-enabled)'),
      '#default_value' => (bool) $this->configuration['enable_pattern'],
      '#description' => $this->t('When enabled, authors can select the Default mode in the element. The Default values support tokens.'),
    ];

    $form['pattern'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default slot ID'),
      '#default_value' => $this->configuration['pattern'],
      '#description' => $this->t('Supports tokens; e.g. slot-[paragraph:parent_entity:entity_type]-[paragraph:parent_entity:type]-[paragraph:delta].'),
      '#states' => [
        'visible' => [
          ':input[name$="[enable_pattern]"]' => ['checked' => TRUE],
        ],
        'required' => [
          ':input[name$="[enable_pattern]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    if ($this->moduleHandler->moduleExists('token')) {
      $form['pattern_tokens'] = [
        '#theme' => 'token_tree_link',
        '#token_types' => ['paragraph', 'node', 'site', 'current-user'],
        '#global_types' => TRUE,
        '#dialog' => TRUE,
        '#show_restricted' => FALSE,
        '#weight' => 1,
        '#states' => [
          'visible' => [
            ':input[name$="[enable_pattern]"]' => ['checked' => TRUE],
          ],
        ],
      ];
    }

    $form['label_pattern'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default slot label (optional)'),
      '#default_value' => $this->configuration['label_pattern'],
      '#description' => $this->t('Optional. Supports tokens; e.g. Autogenerated slot on [paragraph:parent_entity:entity_type] [paragraph:parent_entity:type]'),
      '#states' => [
        'visible' => [
          ':input[name$="[enable_pattern]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    if ($this->moduleHandler->moduleExists('token')) {
      $form['label_tokens'] = [
        '#theme' => 'token_tree_link',
        '#token_types' => ['paragraph', 'node', 'site', 'current-user'],
        '#global_types' => TRUE,
        '#dialog' => TRUE,
        '#show_restricted' => FALSE,
        '#weight' => 1,
        '#states' => [
          'visible' => [
            ':input[name$="[enable_pattern]"]' => ['checked' => TRUE],
          ],
        ],
      ];
    }

    $form['enable_cardinality'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Expose cardinality control'),
      '#default_value' => (bool) $this->configuration['enable_cardinality'],
    ];

    $form['cardinality'] = [
      '#type' => 'number',
      '#title' => $this->t('Default cardinality'),
      '#min' => 0,
      '#default_value' => (int) $this->configuration['cardinality'],
      '#states' => [
        'visible' => [
          ':input[name$="[enable_cardinality]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    $this->configuration['enable_pattern'] = $form_state->getValue('enable_pattern');
    $this->configuration['pattern'] = $form_state->getValue('pattern');
    $this->configuration['enable_cardinality'] = $form_state->getValue('enable_cardinality');
    $this->configuration['cardinality'] = $form_state->getValue('cardinality');
    $this->configuration['label_pattern'] = $form_state->getValue('label_pattern');
  }

  /**
   * {@inheritdoc}
   */
  public function buildBehaviorForm(ParagraphInterface $paragraph, array &$form, FormStateInterface $form_state) {
    $slot_id = $paragraph->getBehaviorSetting($this->getPluginId(), 'slot_id', '');
    $form['slot_selector'] = [
      '#type' => 'slot_selector',
      '#default_value' => $slot_id,
      '#allow_default' => !empty($this->configuration['enable_pattern']),
      '#default_label' => $paragraph->getBehaviorSetting($this->getPluginId(), 'slot_label', ''),
    ];

    // If the previously stored mode was 'default', keep Default selected when
    // editing, even if a default_value exists. Authors without other options
    // rely on this behavior.
    $mode = $paragraph->getBehaviorSetting($this->getPluginId(), 'mode', '');
    if ($mode === 'default') {
      $form['slot_selector']['#default_mode'] = 'default';
    }

    // Optional cardinality control (prefilled from stored setting or default).
    if (!empty($this->configuration['enable_cardinality'])) {
      $cardinality = $paragraph->getBehaviorSetting($this->getPluginId(), 'cardinality', $this->configuration['cardinality']);
      $form['slot_selector']['#cardinality'] = $cardinality;
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitBehaviorForm(ParagraphInterface $paragraph, array &$form, FormStateInterface $form_state) {
    $paragraph->setBehaviorSettings($this->getPluginId(), [
      'mode' => $form_state->getValue(['slot_selector', 'mode']),
      'slot_id' => $form_state->getValue(['slot_selector', 'slot_id']),
      'slot_label' => $form_state->getValue(['slot_selector', 'label']),
      'cardinality' => $form_state->getValue(['slot_selector', 'cardinality']) ?? 0,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(Paragraph $paragraph) {
    $summary = [];

    $mode = $paragraph->getBehaviorSetting($this->getPluginId(), 'mode', '');
    if ($mode !== '') {
      $summary[] = [
        'label' => $this->t('Mode'),
        'value' => $mode,
      ];
    }

    if ($mode !== 'default') {
      $slot_id = $paragraph->getBehaviorSetting($this->getPluginId(), 'slot_id', '');
      $summary[] = [
        'label' => $this->t('Slot id'),
        'value' => $slot_id !== '' ? $slot_id : $this->t('Will be set on save'),
      ];
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function view(array &$build, Paragraph $paragraph, EntityViewDisplayInterface $display, $view_mode) {
    $slot_id = $paragraph->getBehaviorSetting($this->getPluginId(), 'slot_id');
    $slot_label = $paragraph->getBehaviorSetting($this->getPluginId(), 'slot_label', '');

    // Replace tokens if the mode is 'default' (behavior-managed defaults).
    if (
      $this->configuration['enable_pattern']
      && $paragraph->getBehaviorSetting($this->getPluginId(), 'mode', '') === 'default'
    ) {
      $slot_id = $this->token->replace($this->configuration['pattern'], ['paragraph' => $paragraph]);

      if ($this->configuration['label_pattern']) {
        $slot_label = $this->token->replace($this->configuration['label_pattern'], ['paragraph' => $paragraph]);
      }
    }

    if ($slot_id) {
      $build[] = $this->slotsService->buildSlot(
        $slot_id,
        $paragraph->getBehaviorSetting($this->getPluginId(), 'cardinality', 0),
        $slot_label,
      )->toRenderable();
    }
  }

}
