<?php

namespace Drupal\slots_views\Plugin\views\area;

use Drupal\Component\Utility\Html;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\slots\SlotsServiceInterface;
use Drupal\views\Attribute\ViewsArea;
use Drupal\views\Plugin\views\area\TokenizeAreaPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Views area slot handler.
 *
 * @ingroup views_area_handlers
 */
#[ViewsArea("slot")]
class Slot extends TokenizeAreaPluginBase implements ContainerFactoryPluginInterface {

  /**
   * Creates a Slot views area instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\slots\SlotsServiceInterface $slotService
   *   The slots' service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected readonly SlotsServiceInterface $slotService,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('slots.service')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['slot_id'] = ['default' => ''];
    $options['cardinality'] = ['default' => 0];
    $options['label'] = ['default' => ''];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    $form['slot_selector'] = [
      '#type' => 'slot_selector',
      '#default_value' => $this->options['slot_id'],
      '#cardinality' => $this->options['cardinality'],
    ];

    parent::buildOptionsForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitOptionsForm(&$form, FormStateInterface $form_state) {
    parent::submitOptionsForm($form, $form_state);
    $this->options['slot_id'] = $form_state->getValue([
      'options', 'slot_selector', 'slot_id',
    ]);
    $this->options['label'] = $form_state->getValue([
      'options', 'slot_selector', 'label',
    ]);
    $this->options['cardinality'] = $form_state->getValue([
      'options', 'slot_selector', 'cardinality',
    ]) ?? 0;
  }

  /**
   * {@inheritdoc}
   */
  public function render($empty = FALSE) {
    if (!$empty || !empty($this->options['empty'])) {
      $slot_id = $this->tokenizeValue($this->options['slot_id']);
      $slot_id = mb_strtolower(Html::cleanCssIdentifier($slot_id));
      return $this->slotService->buildSlot($slot_id, $this->options['cardinality'], $this->options['label'])
        ->toRenderable();
    }

    return [];
  }

}
