<?php

/**
 * @file
 * Post update functions for Slots.
 */

use Drupal\Core\Entity\Sql\SqlContentEntityStorage;

/**
 * Implements hook_post_update_NAME().
 *
 * Adds a status column to all existing slots fields.
 */
function slots_post_update_add_slots_columns_to_field(&$sandbox) {
  $field_type = 'slots';
  $properties = ['status', 'slots_status'];
  $database_schema = \Drupal::database()->schema();

  $entity_definition_update_manager = \Drupal::entityDefinitionUpdateManager();

  // First, get all fields of the provided field type.
  $field_map = \Drupal::service('entity_field.manager')->getFieldMapByFieldType($field_type);

  // Process per entity type (i.e., node, paragraph) all fields.
  foreach ($field_map as $entity_type_id => $fields) {
    $entity_type_storage = \Drupal::entityTypeManager()->getStorage($entity_type_id);

    // Database schema can only be updated, if the values are stored in the
    // database, of course.
    if ($entity_type_storage instanceof SqlContentEntityStorage) {
      foreach (array_keys($fields) as $field_name) {
        $field_storage_definition = $entity_definition_update_manager->getFieldStorageDefinition($field_name, $entity_type_id);

        // Database column definition, according to each field property.
        $field_columns = $field_storage_definition->getColumns();

        // Get the table and column names, according to the field definition.
        /**
         * @var Drupal\Core\Entity\Sql\DefaultTableMapping $table_mapping
         */
        $table_mapping = $entity_type_storage->getTableMapping([
          $field_name => $field_storage_definition,
        ]);
        $table_names = $table_mapping->getDedicatedTableNames();
        $table_columns = $table_mapping->getColumnNames($field_name);

        // Check if, for the property, the corresponding table exists in the
        // database. If not, create the table column (according to the schema
        // provided by the field definition).
        foreach ($properties as $property) {
          foreach ($table_names as $table_name) {
            $table_exists = $database_schema->tableExists($table_name);
            $field_exists = $database_schema->fieldExists($table_name, $table_columns[$property]);

            if ($table_exists && !$field_exists) {
              $database_schema->addField($table_name, $table_columns[$property], $field_columns[$property]);
            }
          }
        }
        // Reflect any changes made to the field storage definition.
        $entity_definition_update_manager->updateFieldStorageDefinition($field_storage_definition);
      }
    }
  }

  return t('Slots fields have been updated.');
}

/**
 * Implements hook_post_update_NAME().
 *
 * Install the slot entity.
 */
function slots_post_update_install_slot_entity() {
  $slot_entity_type = \Drupal::entityTypeManager()->getDefinition('slot');
  \Drupal::entityDefinitionUpdateManager()->installEntityType($slot_entity_type);

  $slot_ids = \Drupal::service('keyvalue')->get('slot_ids')->getAll();
  $slots_service = \Drupal::service('slots.service');
  $messenger = \Drupal::messenger();
  $logger = \Drupal::logger('slots');
  if (is_iterable($slot_ids)) {
    foreach ($slot_ids as $slot_id) {
      $args = ['%label' => $slot_id];
      if ($slots_service->createSlot($slot_id)) {
        $messenger->addStatus(t('New slot %label has been created.', $args));
        $logger->notice('New slot %label has been created.', $args);
      }
      else {
        $messenger->addError(t('Slot %label could not be converted.', $args));
        $logger->notice('Slot %could not be converted.', $args);
      }
    }
  }

  // Clean up key value store.
  \Drupal::service('keyvalue')->get('slot_ids')->deleteAll();
}
