<?php

declare(strict_types=1);

namespace Drupal\slots\Element;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Attribute\FormElement;
use Drupal\Core\Render\Element\FormElementBase;
use Drupal\slots\SlotIdMachineNameTrait;

/**
 * Provides a reusable element to select or create a slot ID.
 *
 * Properties:
 * - #default_value: The pre-selected slot ID value (string).
 * - #allow_default: When TRUE, shows a "Default" option that delegates value
 *   handling to the consumer (e.g., behavior-managed defaults). The element
 *   itself does not implement token/pattern logic.
 *
 * Behavior:
 * - Shows radios to choose between: Default (optional), Use existing, Create
 *   new.
 * - Sub-elements are shown/hidden via #states.
 * - Return value:
 *   - When the element name is `slot_id` (e.g., Block/Views), it returns a
 *     scalar string (the final slot ID). In `default` mode it returns an empty
 *     string so the consumer can determine the value.
 *   - In other contexts (e.g., Paragraph behavior where the element key is not
 *     `slot_id`), it returns an associative array of its children
 *     (`mode`, `existing_slot_id`, `slot_id`, `label`, `cardinality`).
 */
#[FormElement('slot_selector')]
class SlotSelector extends FormElementBase {

  use SlotIdMachineNameTrait;

  /**
   * {@inheritdoc}
   */
  public function getInfo() {
    $class = static::class;
    return [
      '#input' => TRUE,
      '#tree' => TRUE,
      '#process' => [
        [$class, 'processElement'],
      ],
      '#value_callback' => [$class, 'valueCallback'],
      '#theme_wrappers' => ['form_element'],
      // When TRUE, show a "Default" option that delegates value handling to
      // the consumer (e.g., behavior-managed defaults). No token/pattern logic
      // lives here.
      '#allow_default' => FALSE,
      // Optional default label for the human-readable name field.
      '#default_label' => '',
      // Optional cardinality support; when set (int), a numeric field will be
      // rendered as a sub-element and its value can be accessed from
      // $form_state->getValue([...,'cardinality']).
      '#cardinality' => NULL,
      // Allow callers to explicitly set the default selected mode. When set to
      // 'pattern' and a pattern is available, the element will default to the
      // pattern option even if a #default_value exists.
      '#default_mode' => NULL,
    ];
  }

  /**
   * Process callback to build the element UI.
   */
  public static function processElement(array &$element, FormStateInterface $form_state, array &$complete_form): array {
    $current_user = \Drupal::currentUser();
    $module_handler = \Drupal::moduleHandler();
    /** @var \Drupal\slots\SlotsServiceInterface $slots_service */
    $slots_service = \Drupal::service('slots.service');

    $can_create = $current_user->hasPermission('create slots');
    $allow_default = !empty($element['#allow_default']);

    // Show the Default option first when allowed. This delegates value handling
    // to the consumer (e.g., behavior-managed defaults).
    if ($allow_default) {
      $mode_options['default'] = t('Default');
    }
    // Always allow selecting an existing slot (place it before "Create new").
    $mode_options['existing'] = t('Use existing');
    if ($can_create) {
      $mode_options['new'] = t('Create new');
    }

    $default_value = $element['#default_value'] ?? '';
    $existing_ids = $slots_service->getSlotIds();

    // Honor an explicitly requested default mode from the caller. This is
    // primarily used so that if the author previously chose the Default option,
    // we keep it selected on subsequent edits even if a default value exists.
    $requested_default_mode = $element['#default_mode'] ?? NULL;
    if ($requested_default_mode === 'default' && $allow_default) {
      $default_mode = 'default';
    }
    // Otherwise, if a default value exists, treat it as an existing choice even
    // when the slot entity might not have been created yet (e.g. collected only
    // on render). This avoids prefilling the "Create new" inputs when editing.
    elseif ($default_value) {
      $default_mode = 'existing';
      // Ensure the existing select can display the saved value even if it does
      // not (yet) exist in storage.
      if (!isset($existing_ids[$default_value])) {
        $existing_ids = [$default_value => $default_value] + $existing_ids;
      }
    }
    elseif ($allow_default) {
      $default_mode = 'default';
    }
    else {
      // Fallback to the first available option.
      $default_mode = array_key_first($mode_options);
    }

    $element['mode'] = [
      '#type' => 'radios',
      '#title' => t('Slot selection'),
      '#options' => $mode_options,
      '#default_value' => $form_state->getValue(array_merge($element['#parents'], ['mode'])) ?? $default_mode,
      '#required' => TRUE,
      // Display the radio options horizontally to save vertical space.
      '#attributes' => ['class' => ['container-inline']],
      '#wrapper_attributes' => ['class' => ['container-inline']],
    ];

    // Existing slot selection.
    $select_type = $module_handler->moduleExists('tagify') ? 'select_tagify' : 'select';
    $element['existing_slot_id'] = [
      '#type' => $select_type,
      '#title' => t('Existing slot'),
      '#options' => $existing_ids,
      '#default_value' => isset($existing_ids[$default_value]) ? $default_value : NULL,
      // Required only when using existing mode.
      '#required' => FALSE,
      '#states' => [
        'visible' => [
          ':input[name$="' . static::buildStatesSelectorSuffix($element, 'mode') . '"]' => ['value' => 'existing'],
        ],
        'required' => [
          ':input[name$="' . static::buildStatesSelectorSuffix($element, 'mode') . '"]' => ['value' => 'existing'],
        ],
      ],
    ];

    if ($module_handler->moduleExists('tagify')) {
      $identifier = $complete_form['#id'] ?? implode('-', $element['#parents']);
      $element['existing_slot_id']['#identifier'] = $identifier;
      $element['existing_slot_id']['#attributes']['class'][] = $identifier;
    }

    // Optional human-readable label when creating a new slot.
    // Note: In pattern mode, label comes from the behavior-level label pattern
    // (render-time token replacement), so no input is required.
    // Only prefill the label when creating a new slot. If we are editing an
    // existing configuration (default_value matches an existing slot), keep the
    // label empty to avoid confusing prefilled values when switching modes.
    $prefill_label = (string) ($element['#default_label'] ?? '');
    if ($default_value && isset($existing_ids[$default_value])) {
      $prefill_label = '';
    }
    $element['label'] = [
      '#type' => 'textfield',
      '#title' => t('Slot label'),
      '#maxlength' => 255,
      '#default_value' => $prefill_label,
      '#description' => t('Human-readable name for the slot.'),
      '#required' => FALSE,
      '#states' => [
        'visible' => [
          ':input[name$="' . static::buildStatesSelectorSuffix($element, 'mode') . '"]' => ['value' => 'new'],
        ],
        'required' => [
          ':input[name$="' . static::buildStatesSelectorSuffix($element, 'mode') . '"]' => ['value' => 'new'],
        ],
      ],
    ];

    // Machine name for new slot, sourced from the label above.
    // Wrap the machine_name element in a container so #states reliably hide the
    // entire form item in nested forms (e.g., Paragraphs behavior forms).
    $element['slot_id_container'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ':input[name$="' . static::buildStatesSelectorSuffix($element, 'mode') . '"]' => ['value' => 'new'],
        ],
      ],
    ];
    $element['slot_id_container']['slot_id'] = [
      '#type' => 'machine_name',
      '#title' => t('New slot ID'),
      '#maxlength' => 255,
      '#default_value' => !$default_value || isset($existing_ids[$default_value]) ? '' : $default_value,
      '#required' => FALSE,
      // Ensure the input name remains <parents>[slot_id] so valueCallback keeps
      // working.
      '#parents' => array_merge($element['#parents'], ['slot_id']),
      '#states' => [
        'required' => [
          ':input[name$="' . static::buildStatesSelectorSuffix($element, 'mode') . '"]' => ['value' => 'new'],
        ],
      ],
      '#machine_name' => [
        'exists' => [static::class, 'slotIdExists'],
        'replace_pattern' => static::getReplacePattern(),
        'error' => t('The slot ID must contain only lowercase letters, numbers, underscores and hyphens.'),
        // Use the label field as the source for automatic ID suggestion.
        'source' => array_merge($element['#array_parents'] ?? $element['#parents'], ['label']),
      ],
    ];

    // Optional cardinality control.
    $cardinality_default = $element['#cardinality'] ?? NULL;
    if ($cardinality_default !== NULL) {
      $element['cardinality'] = [
        '#type' => 'number',
        '#title' => t('Slot cardinality'),
        '#description' => t('Set to 0 for unlimited items.'),
        '#min' => 0,
        '#default_value' => (int) $cardinality_default,
      ];
    }
    else {
      $element['cardinality'] = [
        '#type' => 'hidden',
        '#value' => 0,
      ];
    }

    // Attach light CSS tweaks for admin theme spacing (e.g., Gin).
    $element['#attached']['library'][] = 'slots/drupal.slots.slot_selector';

    return $element;
  }

  /**
   * Value callback to return the final slot_id.
   */
  public static function valueCallback(&$element, $input, FormStateInterface $form_state) {
    // If no input yet, return the default value for initial rendering.
    if ($input === FALSE) {
      return $element['#default_value'] ?? '';
    }

    switch ($input['mode']) {
      case 'default':
        $input['slot_id'] = '';
        $input['label'] = '';
        break;

      case 'existing':
        $input['slot_id'] = $input['existing_slot_id'];
        $input['label'] = '';
        break;

      case 'new':
        break;
    }

    return [
      'mode' => $input['mode'],
      'slot_id' => $input['slot_id'],
      'label' => $input['label'] ?? '',
      'cardinality' => isset($input['cardinality']) ? (int) $input['cardinality'] : 0,
    ];
  }

  /**
   * Helper to build a robust ends-with suffix for #states selectors.
   *
   * Generates a suffix like '[<last-parent>][<child>]' or '[<child>]' when
   * there is no parent. This works with ':input[name$="<suffix>"]' reliably
   * in nested admin forms (e.g., Paragraphs behavior config forms).
   */
  protected static function buildStatesSelectorSuffix(array $element, string $child): string {
    $parents = $element['#parents'];
    $parts = [];
    if (!empty($parents)) {
      $parts[] = $parents[count($parents) - 1];
    }
    $parts[] = $child;
    return '[' . implode('][', $parts) . ']';
  }

  /**
   * Machine name exists callback; slots can be non-unique, always FALSE.
   */
  public static function slotIdExists(): bool {
    return FALSE;
  }

}
