<?php

declare(strict_types=1);

namespace Drupal\slots\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\slots\Plugin\Field\ComputedSlotContentFieldItemList;
use Drupal\slots\SlotInterface;

/**
 * Defines the slot entity class.
 *
 * @ContentEntityType(
 *   id = "slot",
 *   label = @Translation("Slot"),
 *   label_collection = @Translation("Slots"),
 *   label_singular = @Translation("slot"),
 *   label_plural = @Translation("slots"),
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "langcode" = "langcode",
 *     "uuid" = "uuid",
 *   },
 *   handlers = {
 *     "list_builder" = "Drupal\slots\SlotListBuilder",
 *     "views_data" = "Drupal\slots\SlotsViewsData",
 *     "form" = {
 *       "add" = "Drupal\slots\Form\SlotForm",
 *       "edit" = "Drupal\slots\Form\SlotForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *       "delete-multiple-confirm" = "Drupal\Core\Entity\Form\DeleteMultipleForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\slots\Routing\SlotHtmlRouteProvider",
 *     },
 *   },
 *   links = {
 *     "add-form" = "/admin/content/slots/add",
 *     "canonical" = "/admin/content/slots/{slot}",
 *     "edit-form" = "/admin/content/slots/{slot}",
 *     "delete-form" = "/admin/content/slots/{slot}/delete",
 *     "delete-multiple-form" = "/admin/content/slot/delete-multiple",
 *   },
 *   admin_permission = "administer slots",
 *   base_table = "slot",
 *   data_table = "slot_field_data",
 *   translatable = TRUE,
 *   label_count = {
 *     "singular" = "@count slot",
 *     "plural" = "@count slots",
 *   }
 * )
 */
final class Slot extends ContentEntityBase implements SlotInterface {

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields[$entity_type->getKey('id')] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('ID'))
      ->setReadOnly(TRUE)
      ->setSetting('unsigned', TRUE);

    $fields['label'] = BaseFieldDefinition::create('string')
      ->setTranslatable(TRUE)
      ->setLabel(t('Label'))
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Authored on'))
      ->setTranslatable(FALSE)
      ->setDescription(t('The time that the slot was created.'))
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', FALSE);

    $fields['content'] = BaseFieldDefinition::create('dynamic_entity_reference')
      ->setLabel(new TranslatableMarkup('Slot content'))
      ->setDescription(t('The content used in this slot.'))
      ->setComputed(TRUE)
      ->setComputed(TRUE)
      ->setClass(ComputedSlotContentFieldItemList::class)
      ->setInternal(FALSE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'dynamic_entity_reference_label',
        'settings' => [
          'link' => TRUE,
        ],
      ])
      ->setDisplayOptions('form', [
        'type' => 'slot_content_weights',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('view', TRUE);

    return $fields;
  }

}
