<?php

declare(strict_types=1);

namespace Drupal\slots\Form;

use Drupal\Core\Url;
use Drupal\Core\Form\FormStateInterface;
use Drupal\slots\SlotIdMachineNameTrait;
use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Routing\RouteProviderInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form controller for the slot entity edit forms.
 */
final class SlotForm extends ContentEntityForm {

  use SlotIdMachineNameTrait;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity.repository'),
      $container->get('entity_type.bundle.info'),
      $container->get('datetime.time'),
      $container->get('router.route_provider'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function __construct(
    EntityRepositoryInterface $entity_repository,
    EntityTypeBundleInfoInterface $entity_type_bundle_info,
    TimeInterface $time,
    protected RouteProviderInterface $routeProvider,
  ) {
    parent::__construct($entity_repository, $entity_type_bundle_info, $time);
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);
    $form['id'] = [
      '#type' => 'machine_name',
      '#maxlength' => 64,
      '#description' => $this->t('A unique name for this slot. Must be alpha-numeric and underscore or hyphens separated.'),
      '#default_value' => !$this->entity->isNew() ? $this->entity->id() : '',
      '#machine_name' => [
        'exists' => [$this, 'slotIdExistsCallback'],
        'replace_pattern' => $this->getReplacePattern(),
        'error' => $this->t('The slot ID name must contain only lowercase letters, numbers, underscores and hyphens.'),
        'source' => ['label', 'widget', 0, 'value'],
      ],
      '#required' => TRUE,
      '#disabled' => !$this->entity->isNew(),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    $result = parent::save($form, $form_state);

    $message_args = ['%label' => $this->entity->toLink()->toString()];
    $logger_args = [
      '%label' => $this->entity->label(),
      'link' => $this->entity->toLink($this->t('View'))->toString(),
    ];

    switch ($result) {
      case SAVED_NEW:
        $this->messenger()->addStatus($this->t('New slot %label has been created.', $message_args));
        $this->logger('slots')->notice('New slot %label has been created.', $logger_args);
        break;

      case SAVED_UPDATED:
        $this->messenger()->addStatus($this->t('The slot %label has been updated.', $message_args));
        $this->logger('slots')->notice('The slot %label has been updated.', $logger_args);
        break;

      default:
        throw new \LogicException('Could not save the entity.');
    }

    // If the slot_content view is present use it as default destination.
    $slots_content_route_name = 'view.slot_content.page';
    if (!empty($this->routeProvider->getRoutesByNames([$slots_content_route_name]))) {
      $form_state->setRedirectUrl(Url::fromRoute($slots_content_route_name));
    }

    return $result;
  }

}
