<?php

declare(strict_types=1);

namespace Drupal\slots\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Block\Attribute\Block;
use Drupal\slots\SlotsServiceInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\RedirectDestinationInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\block_plugin_view_builder\BlockPluginViewBuilderInterface;

/**
 * Provides a slot block.
 */
#[Block(
  id: "slot_block",
  admin_label: new TranslatableMarkup("Slot block"),
)]
class SlotBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * Creates a SlotBlock instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\slots\SlotsServiceInterface $slotsService
   *   The slots service.
   * @param \Drupal\block_plugin_view_builder\BlockPluginViewBuilderInterface $blockPluginViewBuilder
   *   The block plugin view builder service.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Routing\RedirectDestinationInterface $redirectDestination
   *   The redirect destination service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected readonly SlotsServiceInterface $slotsService,
    protected readonly BlockPluginViewBuilderInterface $blockPluginViewBuilder,
    protected readonly AccountInterface $currentUser,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly RedirectDestinationInterface $redirectDestination,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('slots.service'),
      $container->get('block_plugin.view_builder'),
      $container->get('current_user'),
      $container->get('entity_type.manager'),
      $container->get('redirect.destination'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return array_merge(parent::defaultConfiguration(), [
      'label' => '',
      'slot_label' => '',
      'slot_id' => '',
      'cardinality' => 0,
      'empty' => FALSE,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    // Remove label display settings, since this block is only used as a
    // container.
    $form['label_display']['#default_value'] = FALSE;
    $form['label_display']['#access'] = FALSE;

    $form['slot_selector'] = [
      '#type' => 'slot_selector',
      '#default_value' => $this->configuration['slot_id'],
      '#cardinality' => $this->configuration['cardinality'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    parent::blockSubmit($form, $form_state);

    $this->configuration['slot_id'] = $form_state->getValue([
      'slot_selector', 'slot_id',
    ]);
    $this->configuration['slot_label'] = $form_state->getValue([
      'slot_selector', 'label',
    ]);
    $this->configuration['cardinality'] = $form_state->getValue([
      'slot_selector', 'cardinality',
    ]) ?? 0;

    // Create or update the slot with the provided label.
    $this->slotsService->createSlot($this->configuration['slot_id'], $this->configuration['slot_label']);
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $build = [];

    $slot_label = (string) ($this->configuration['slot_label'] ?? $this->configuration['label'] ?? '');
    $this->slotsService->createSlot($this->configuration['slot_id'], $slot_label);

    if (!$this->configuration['empty']) {
      $build = $this->slotsService->buildSlotContents($this->configuration['slot_id']);

      // Enforce cardinality: if a limit is set (> 0), only render up to that
      // many slot contents. Any additional matching contents are ignored.
      $cardinality = (int) ($this->configuration['cardinality'] ?? 0);
      if ($cardinality > 0 && (!empty($build['#length']) && $build['#length'] > $cardinality)) {
        $kept = 0;
        foreach ($build as $key => $item) {
          // Skip non-content metadata and properties.
          if (str_starts_with((string) $key, '#')) {
            continue;
          }
          // Count only actual content entries.
          if (is_array($item) && array_key_exists('content', $item)) {
            $kept++;
            if ($kept > $cardinality) {
              unset($build[$key]);
            }
          }
        }
        $build['#length'] = $cardinality;
      }

      if ($this->configuration['cardinality'] > $build['#length'] || $this->configuration['cardinality'] == 0) {
        $build[] = $this->blockPluginViewBuilder->view('slot_block', [
          'slot_id' => $this->configuration['slot_id'],
          'empty' => TRUE,
        ]);
      }

      $build['#attributes']['class'][] = 'is-slot-wrapper';
    }
    else {
      if ($this->currentUser->hasPermission('view slot identifiers')) {
        $slot = $this->entityTypeManager->getStorage('slot')->load($this->configuration['slot_id']);
        $build[] = [
          '#type' => 'inline_template',
          '#template' => '<div class="slots--slot__identifier">{{ content }}</div>',
          '#context' => [
            'content' => $this->t('Slot: @slot_label (<a href=":edit-url">edit</a>)', [
              ':edit-url' => $slot->toUrl('edit-form', [
                'query' => $this->redirectDestination->getAsArray(),
              ])->toString(),
              '@slot_label' => $slot->label(),
            ]),
          ],
        ];
        $this->configuration['empty'] = TRUE;
        $build['#attributes']['class'][] = 'is-empty-slot';
      }
    }

    if (!empty($build)) {
      $build['#attributes']['class'][] = 'slots--slot';
      $build['#attached']['library'][] = 'slots/drupal.slots.theme';
    }

    return $build;
  }

}
